# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Example: positions = {'drawer': {...}, 'rubbish': {...}, 'bin': {...}, ...}

        # --- Exploration Phase: Identify Missing Predicate ---
        # Based on feedback, we need to check for missing predicates such as lock-known, is-locked, or object location.
        # We'll perform exploration to check the state of the drawer and the object.

        # For demonstration, let's assume the following keys exist in positions:
        # 'drawer', 'rubbish', 'bin', 'gripper', and each has a dict with relevant info.

        # 1. Check if the drawer is locked (simulate exploration)
        drawer_info = positions.get('drawer', {})
        rubbish_info = positions.get('rubbish', {})
        bin_info = positions.get('bin', {})
        gripper_info = positions.get('gripper', {})

        # If the drawer's lock state is unknown, try to pull to learn it (simulate exploration)
        is_locked = drawer_info.get('is_locked', None)
        if is_locked is None:
            print("[Exploration] Drawer lock state unknown. Attempting to pull to learn lock state.")
            try:
                # Try to pull the drawer to learn its lock state
                obs, reward, done = pull(env, task, drawer_info)
                # After pull, update is_locked if possible
                positions = get_object_positions()
                drawer_info = positions.get('drawer', {})
                is_locked = drawer_info.get('is_locked', None)
            except Exception as e:
                print(f"[Exploration] Exception during pull: {e}")

        if is_locked:
            print("[Exploration] Drawer is locked. Cannot proceed with opening.")
            # Depending on the task, you may abort or try to unlock (if skill exists)
            return

        # 2. Check if the rubbish is in the correct location before moving it
        rubbish_loc = rubbish_info.get('location', None)
        if rubbish_loc != drawer_info.get('location', None):
            print(f"[Exploration] Rubbish is not in the drawer. Current location: {rubbish_loc}")
            # If not in the drawer, cannot proceed with pick/place
            return

        # 3. Check gripper position and orientation before manipulation
        gripper_pos = gripper_info.get('position', None)
        gripper_quat = gripper_info.get('quaternion', None)
        required_pos = drawer_info.get('handle_position', None)
        required_quat = drawer_info.get('handle_quaternion', None)
        if gripper_pos is not None and required_pos is not None:
            pos_dist = np.linalg.norm(np.array(gripper_pos) - np.array(required_pos))
            if pos_dist > 0.05:
                print(f"[Exploration] Gripper not at required position for drawer handle. Moving gripper.")
                try:
                    obs, reward, done = move(env, task, gripper_pos, required_pos)
                except Exception as e:
                    print(f"[Exploration] Exception during move: {e}")
        if gripper_quat is not None and required_quat is not None:
            quat_diff = np.dot(np.array(gripper_quat), np.array(required_quat))
            if quat_diff < 0.99:
                print(f"[Exploration] Gripper not at required orientation. Rotating gripper.")
                try:
                    obs, reward, done = rotate(env, task, required_quat)
                except Exception as e:
                    print(f"[Exploration] Exception during rotate: {e}")

        # --- Main Task Plan Execution ---
        # 1. Rotate gripper to required orientation (if not already done)
        if gripper_quat is not None and required_quat is not None:
            quat_diff = np.dot(np.array(gripper_quat), np.array(required_quat))
            if quat_diff < 0.99:
                print("[Task] Rotating gripper to required orientation for drawer.")
                try:
                    obs, reward, done = rotate(env, task, required_quat)
                    if done:
                        print("[Task] Task ended during rotate!")
                        return
                except Exception as e:
                    print(f"[Task] Exception during rotate: {e}")
                    return

        # 2. Move gripper to drawer handle position (if not already done)
        if gripper_pos is not None and required_pos is not None:
            pos_dist = np.linalg.norm(np.array(gripper_pos) - np.array(required_pos))
            if pos_dist > 0.05:
                print("[Task] Moving gripper to drawer handle position.")
                try:
                    obs, reward, done = move(env, task, gripper_pos, required_pos)
                    if done:
                        print("[Task] Task ended during move!")
                        return
                except Exception as e:
                    print(f"[Task] Exception during move: {e}")
                    return

        # 3. Pull the drawer open
        print("[Task] Pulling the drawer open.")
        try:
            obs, reward, done = pull(env, task, drawer_info)
            if done:
                print("[Task] Task ended during pull!")
                return
        except Exception as e:
            print(f"[Task] Exception during pull: {e}")
            return

        # 4. Pick the rubbish from the drawer
        print("[Task] Picking the rubbish from the drawer.")
        try:
            obs, reward, done = pick(env, task, rubbish_info, drawer_info)
            if done:
                print("[Task] Task ended during pick!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick: {e}")
            return

        # 5. Move to the bin location
        rubbish_current_loc = rubbish_info.get('location', None)
        bin_loc = bin_info.get('location', None)
        if rubbish_current_loc != bin_loc:
            print("[Task] Moving to bin location.")
            try:
                obs, reward, done = move(env, task, rubbish_current_loc, bin_loc)
                if done:
                    print("[Task] Task ended during move to bin!")
                    return
            except Exception as e:
                print(f"[Task] Exception during move to bin: {e}")
                return

        # 6. Place the rubbish in the bin
        print("[Task] Placing the rubbish in the bin.")
        try:
            obs, reward, done = place(env, task, rubbish_info, bin_info)
            if done:
                print("[Task] Task ended after placing rubbish in bin!")
                return
        except Exception as e:
            print(f"[Task] Exception during place: {e}")
            return

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()