# run_skeleton_task.py (Completed with Exploration and Error Handling)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

import time

def euler_from_quat(quaternion):
    """
    Convert a quaternion to Euler angles (ZYX order).
    """
    # Assumes quaternion is [x, y, z, w]
    x, y, z, w = quaternion
    t0 = +2.0 * (w * x + y * z)
    t1 = +1.0 - 2.0 * (x * x + y * y)
    X = np.arctan2(t0, t1)
    t2 = +2.0 * (w * y - z * x)
    t2 = np.clip(t2, -1.0, 1.0)
    Y = np.arcsin(t2)
    t3 = +2.0 * (w * z + x * y)
    t4 = +1.0 - 2.0 * (y * y + z * z)
    Z = np.arctan2(t3, t4)
    return np.array([X, Y, Z])

def normalize_quaternion(quaternion):
    """
    Normalize a quaternion.
    """
    q = np.array(quaternion)
    norm = np.linalg.norm(q)
    if norm == 0:
        return q
    return q / norm

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # We need to check for missing predicates (e.g., lock-known, is-locked, etc.)
        # and ensure we know the state of the drawer before acting.

        # Let's assume we are looking for a drawer to open and an object (rubbish) to pick and place.
        # We'll use the available skills: ['pick', 'place', 'move', 'rotate', 'pull']

        # For demonstration, let's try to find a drawer and a rubbish object.
        drawer_name = None
        rubbish_name = None
        for name in positions:
            if 'drawer' in name.lower():
                drawer_name = name
            if 'rubbish' in name.lower() or 'trash' in name.lower() or 'object' in name.lower():
                rubbish_name = name
        if drawer_name is None:
            print("[Error] No drawer found in the environment!")
            return
        if rubbish_name is None:
            print("[Error] No rubbish object found in the environment!")
            return

        print(f"[Info] Target drawer: {drawer_name}")
        print(f"[Info] Target rubbish: {rubbish_name}")

        # Check if the drawer is locked (simulate missing predicate knowledge)
        # In a real system, we might have a predicate like is-locked or lock-known.
        # Here, we simulate exploration: try to pull, and if it fails, assume it's locked.

        # 1. Move to the drawer position (if move skill is available)
        try:
            drawer_pos = positions[drawer_name]
            print(f"[Action] Moving to drawer at {drawer_pos}")
            obs, reward, done = move(env, task, target_pos=drawer_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
            if done:
                print("[Task] Task ended after move to drawer!")
                return
        except Exception as e:
            print(f"[Warning] Move to drawer failed: {e}")

        # 2. Rotate gripper if needed (simulate opening handle)
        try:
            # For demonstration, let's rotate to 90 degrees (quaternion for 90 deg about z)
            target_quat = [0, 0, np.sin(np.pi/4), np.cos(np.pi/4)]  # [x, y, z, w]
            print(f"[Action] Rotating gripper to 90 degrees (quaternion: {target_quat})")
            obs, reward, done = rotate(env, task, target_quat, max_steps=100, threshold=0.05, timeout=10.0)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Warning] Rotate failed: {e}")

        # 3. Try to pull the drawer (simulate exploration for lock-known/is-locked)
        drawer_unlocked = False
        try:
            print(f"[Action] Attempting to pull the drawer: {drawer_name}")
            obs, reward, done = pull(env, task, drawer_name)
            drawer_unlocked = True
            print("[Info] Drawer pulled successfully (assume unlocked).")
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[Warning] Pull failed: {e}")
            print("[Exploration] Drawer may be locked or lock state unknown. Marking as locked.")
            drawer_unlocked = False

        if not drawer_unlocked:
            print("[Exploration] Attempting to discover lock state (missing predicate: lock-known/is-locked).")
            # Here, in a real exploration, we might try to sense or perform another action.
            # For now, we log and skip further drawer actions.
            print("[Error] Drawer is locked or lock state unknown. Cannot proceed to open.")
            return

        # 4. Move to rubbish object
        try:
            rubbish_pos = positions[rubbish_name]
            print(f"[Action] Moving to rubbish at {rubbish_pos}")
            obs, reward, done = move(env, task, target_pos=rubbish_pos, approach_distance=0.15, max_steps=100, threshold=0.01, approach_axis='z', timeout=10.0)
            if done:
                print("[Task] Task ended after move to rubbish!")
                return
        except Exception as e:
            print(f"[Warning] Move to rubbish failed: {e}")

        # 5. Pick the rubbish
        try:
            print(f"[Action] Picking up rubbish: {rubbish_name}")
            obs, reward, done = pick(env, task, rubbish_name)
            if done:
                print("[Task] Task ended after pick!")
                return
        except Exception as e:
            print(f"[Warning] Pick failed: {e}")
            print("[Error] Could not pick up rubbish. Aborting.")
            return

        # 6. Place the rubbish in the drawer (assume inside drawer position is known)
        try:
            # For demonstration, place at the drawer's position
            print(f"[Action] Placing rubbish: {rubbish_name} into drawer: {drawer_name}")
            obs, reward, done = place(env, task, rubbish_name, drawer_name)
            if done:
                print("[Task] Task ended after place!")
                return
        except Exception as e:
            print(f"[Warning] Place failed: {e}")
            print("[Error] Could not place rubbish in drawer. Aborting.")
            return

        print("[Task] Oracle plan executed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()