# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback, we need to explore the environment to determine which predicate is missing.
        # We'll use the available skills to interact with objects and locations, and log the results.

        # For demonstration, let's assume we have a list of objects and locations from positions
        # and that the robot is named 'robot1' (if needed for move skill).
        # We'll attempt to move to each location, pick each object, and pull if possible,
        # and log any unexpected failures or missing information.

        # Extract object and location names from positions
        object_names = []
        location_names = set()
        for name, pos in positions.items():
            if name.startswith('obj') or name.startswith('object') or name.startswith('drawer'):
                object_names.append(name)
            if isinstance(pos, dict) and 'location' in pos:
                location_names.add(pos['location'])
            elif isinstance(pos, (tuple, list)) and len(pos) == 3:
                # Heuristic: treat as location if not object
                location_names.add(name)
        location_names = list(location_names)
        if not location_names:
            # Fallback: use some default locations if not found
            location_names = ['location1', 'location2']

        # For exploration, try to move to each location and pick each object
        for loc in location_names:
            try:
                print(f"[Exploration] Attempting to move to location: {loc}")
                # The move skill may require from_location and to_location; we use a dummy 'current' location
                # If not available, skip move
                obs, reward, done = move(env, task, from_location=None, to_location=loc)
                if done:
                    print(f"[Exploration] Task ended after move to {loc}!")
                    return
            except Exception as e:
                print(f"[Exploration] Move to {loc} failed: {e}")

        for obj in object_names:
            # Try to pick the object at its location
            obj_pos = positions.get(obj, None)
            if obj_pos is None:
                print(f"[Exploration] No position found for object {obj}, skipping pick.")
                continue
            # Try to find a location for the object
            obj_loc = None
            if isinstance(obj_pos, dict) and 'location' in obj_pos:
                obj_loc = obj_pos['location']
            elif isinstance(obj_pos, (tuple, list)) and len(obj_pos) == 3:
                # Heuristic: treat as location name if available
                obj_loc = obj
            else:
                obj_loc = None
            if obj_loc is None:
                print(f"[Exploration] No location found for object {obj}, skipping pick.")
                continue
            try:
                print(f"[Exploration] Attempting to pick object: {obj} at location: {obj_loc}")
                obs, reward, done = pick(env, task, obj, obj_loc)
                if done:
                    print(f"[Exploration] Task ended after pick of {obj}!")
                    return
            except Exception as e:
                print(f"[Exploration] Pick of {obj} at {obj_loc} failed: {e}")

        # Try to pull each object (if applicable)
        for obj in object_names:
            obj_pos = positions.get(obj, None)
            obj_loc = None
            if isinstance(obj_pos, dict) and 'location' in obj_pos:
                obj_loc = obj_pos['location']
            elif isinstance(obj_pos, (tuple, list)) and len(obj_pos) == 3:
                obj_loc = obj
            else:
                obj_loc = None
            if obj_loc is None:
                continue
            try:
                print(f"[Exploration] Attempting to pull object: {obj} at location: {obj_loc}")
                obs, reward, done = pull(env, task, obj, obj_loc)
                if done:
                    print(f"[Exploration] Task ended after pull of {obj}!")
                    return
            except Exception as e:
                print(f"[Exploration] Pull of {obj} at {obj_loc} failed: {e}")

        # Try to rotate the gripper to a default orientation (if applicable)
        try:
            print("[Exploration] Attempting to rotate gripper to default orientation.")
            # Use a default quaternion (identity) for demonstration
            target_quat = np.array([0, 0, 0, 1])
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Exploration] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Exploration] Rotate failed: {e}")

        # Try to place each object at each location (if possible)
        for obj in object_names:
            for loc in location_names:
                try:
                    print(f"[Exploration] Attempting to place object: {obj} at location: {loc}")
                    obs, reward, done = place(env, task, obj, loc)
                    if done:
                        print(f"[Exploration] Task ended after place of {obj} at {loc}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Place of {obj} at {loc} failed: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicates or unexpected failures.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()