# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (rotated gripper1 angle_ninety) is missing
        # We need to explore the environment to determine if the gripper is rotated to the required angle

        # Try to check if the gripper is rotated to angle_ninety (or angle_ninety_deg)
        # If not, use the rotate skill to set it

        # For this example, we assume the gripper is named 'gripper1' and angles are 'zero_deg', 'ninety_deg', etc.
        # The actual names may differ in your environment; adjust as needed.

        # Get gripper and angle names from positions or use defaults
        gripper_name = None
        angle_ninety = None
        # Try to infer from positions or use defaults
        for key in positions.keys():
            if 'gripper' in key:
                gripper_name = key
            if 'ninety' in key or '90' in key:
                angle_ninety = key
        if gripper_name is None:
            gripper_name = 'gripper1'
        if angle_ninety is None:
            angle_ninety = 'ninety_deg'

        # Try to get current gripper rotation from observation
        obs = task.get_observation()
        current_angle = None
        if hasattr(obs, 'gripper_angle'):
            current_angle = obs.gripper_angle
        elif hasattr(obs, 'gripper_rot'):
            current_angle = obs.gripper_rot
        else:
            # Fallback: try to get from positions dictionary
            for k in positions.keys():
                if 'angle' in k and positions[k] is not None:
                    current_angle = k
                    break

        # If current_angle is not 'ninety_deg', rotate
        need_rotate = True
        if current_angle is not None:
            if isinstance(current_angle, str):
                if 'ninety' in current_angle or '90' in current_angle:
                    need_rotate = False
            elif isinstance(current_angle, (int, float)):
                # If angle is in degrees or radians, check if close to 90 deg
                if abs(float(current_angle) - 90.0) < 5.0:
                    need_rotate = False

        if need_rotate:
            print(f"[Exploration] Rotating {gripper_name} to {angle_ninety} (missing predicate: rotated {gripper_name} {angle_ninety})")
            try:
                # Try to get current angle name
                from_angle = None
                for k in positions.keys():
                    if 'angle' in k and k != angle_ninety:
                        from_angle = k
                        break
                if from_angle is None:
                    from_angle = 'zero_deg'
                obs, reward, done = rotate(env, task, gripper_name, from_angle, angle_ninety)
                if done:
                    print("[Exploration] Task ended during rotate!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during rotate: {e}")

        else:
            print(f"[Exploration] Gripper {gripper_name} already rotated to {angle_ninety}.")

        # === Main Task Plan ===
        # At this point, the missing predicate (rotated gripper1 angle_ninety) should be satisfied.
        # You can now proceed with the rest of your oracle plan using only the available skills.

        # Example: (This is a placeholder; replace with your actual plan steps)
        # Suppose you want to pick an object and place it somewhere

        # Find an object and a location to place
        object_to_pick = None
        place_location = None
        for k in positions.keys():
            if 'object' in k or 'ball' in k or 'tomato' in k:
                object_to_pick = k
            if 'bin' in k or 'goal' in k or 'target' in k:
                place_location = k
        if object_to_pick is None:
            # Fallback: pick any object
            for k in positions.keys():
                if k != gripper_name and 'drawer' not in k and 'angle' not in k:
                    object_to_pick = k
                    break
        if place_location is None:
            # Fallback: pick any location different from object
            for k in positions.keys():
                if k != object_to_pick and k != gripper_name and 'angle' not in k:
                    place_location = k
                    break

        # Pick the object
        if object_to_pick is not None:
            print(f"[Task] Picking object: {object_to_pick}")
            try:
                obj_pos = positions[object_to_pick]
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after pick!")
                    return
            except Exception as e:
                print(f"[Task] Exception during pick: {e}")

        # Place the object
        if object_to_pick is not None and place_location is not None:
            print(f"[Task] Placing object: {object_to_pick} at {place_location}")
            try:
                place_pos = positions[place_location]
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(place_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after place!")
                    return
            except Exception as e:
                print(f"[Task] Exception during place: {e}")

        # You can add more plan steps here using only the available skills: move, pick, place, rotate, pull

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()