# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Object List Consistency Check ===
        # Ensure the declared object list matches the environment
        try:
            env_object_list = getattr(env, 'object_list', None)
            if env_object_list is not None:
                for obj_name in positions.keys():
                    if obj_name not in env_object_list:
                        print(f"[Warning] Object '{obj_name}' in positions but not in env.object_list.")
        except Exception as e:
            print(f"[Error] Could not check object list consistency: {e}")

        # === Force Calibration (if available) ===
        # Some environments may provide a calibration method for the gripper
        try:
            if hasattr(env, 'gripper_calibrate_forces'):
                env.gripper_calibrate_forces()
                print("[Info] Gripper force calibration complete.")
        except Exception as e:
            print(f"[Error] Gripper force calibration failed: {e}")

        # === Safety Checks ===
        # Check gripper state before starting
        try:
            if hasattr(env, 'is_gripper_open'):
                if not env.is_gripper_open():
                    print("[Safety] Gripper is not open at start. Attempting to open.")
                    if hasattr(env, 'open_gripper'):
                        env.open_gripper()
        except Exception as e:
            print(f"[Error] Safety check failed: {e}")

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback and exploration knowledge suggest that the robot may need to discover a missing predicate (e.g., lock-known, durability-known, etc.)
        # We'll attempt to explore by using available skills and checking for unexpected failures or missing information.

        # For demonstration, we will:
        # 1. Move to each object's location (if move is available)
        # 2. Try to pick each object (if pick is available)
        # 3. Try to pull (if pull is available) and see if any predicate is missing

        # Get available skill names
        available_skills = ['pick', 'place', 'move', 'rotate', 'pull']

        # For each object in the environment, attempt exploration
        for obj_name, obj_pos in positions.items():
            print(f"\n[Exploration] Attempting to interact with object: {obj_name}")

            # 1. Move to object location (if move skill is available)
            if 'move' in available_skills:
                try:
                    print(f"[Exploration] Moving to {obj_name} at position {obj_pos}")
                    # The move skill may require from_pos and to_pos; here we use current gripper position as from_pos
                    obs = task.get_observation()
                    gripper_pos = obs.gripper_pose[:3]
                    # If move expects names, use obj_name; if positions, use obj_pos
                    move(env, task, from_pos=gripper_pos, to_pos=obj_pos)
                except Exception as e:
                    print(f"[Error] Move to {obj_name} failed: {e}")

            # 2. Try to pick the object (if pick skill is available)
            if 'pick' in available_skills:
                try:
                    print(f"[Exploration] Attempting to pick {obj_name}")
                    obs, reward, done = pick(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after picking {obj_name}!")
                        return
                except Exception as e:
                    print(f"[Error] Pick {obj_name} failed: {e}")

            # 3. Try to pull the object (if pull skill is available)
            if 'pull' in available_skills:
                try:
                    print(f"[Exploration] Attempting to pull {obj_name}")
                    # For pull, we need to decide a direction and distance; here we use a default
                    obs, reward, done = pull(
                        env,
                        task,
                        pull_distance=0.1,
                        pull_axis='x',
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after pulling {obj_name}!")
                        return
                except Exception as e:
                    print(f"[Error] Pull {obj_name} failed: {e}")

            # 4. Place the object back (if place skill is available)
            if 'place' in available_skills:
                try:
                    print(f"[Exploration] Attempting to place {obj_name} at original position")
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=obj_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after placing {obj_name}!")
                        return
                except Exception as e:
                    print(f"[Error] Place {obj_name} failed: {e}")

            # 5. Try to rotate (if rotate skill is available)
            if 'rotate' in available_skills:
                try:
                    print(f"[Exploration] Attempting to rotate at {obj_name}")
                    # For rotate, we need to specify from_angle and to_angle; use default values
                    obs = task.get_observation()
                    # Example: rotate from 0 to 90 degrees (in radians)
                    from_angle = 0.0
                    to_angle = np.pi / 2
                    rotate(env, task, from_angle=from_angle, to_angle=to_angle)
                except Exception as e:
                    print(f"[Error] Rotate at {obj_name} failed: {e}")

        print("\n[Exploration] Completed exploration of all objects.")

        # === End of Exploration Phase ===

        # At this point, based on the feedback and exploration, you should be able to identify if any predicate is missing
        # (e.g., if a skill fails due to a missing lock-known, durability-known, etc., you will see the error above)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
