# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates a missing predicate is causing planning to fail.
        # We will use the available skills to explore the environment and infer which predicate is missing.
        # The exploration domain suggests predicates like identified, temperature-known, weight-known, durability-known, lock-known.
        # We will attempt to use the available skills to trigger any observable state changes or errors.

        # For demonstration, we will attempt to:
        # 1. Move to each object location (if possible)
        # 2. Pick each object (if possible)
        # 3. Place each object (if possible)
        # 4. Try to rotate and pull (if possible)
        # Any failure or unexpected result may indicate a missing predicate.

        # Note: Only use the available skills: ['pick', 'place', 'move', 'rotate', 'pull']

        # List of objects and their positions
        object_names = list(positions.keys())

        # Track which predicates might be missing based on action outcomes
        missing_predicates = set()

        # Attempt to move to each object's position
        for obj_name in object_names:
            obj_pos = positions[obj_name]
            print(f"[Exploration] Attempting to move to {obj_name} at {obj_pos}")
            try:
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after moving to {obj_name}!")
                    break
            except Exception as e:
                print(f"[Exploration] Exception during move to {obj_name}: {e}")
                missing_predicates.add('robot-at or at')

        # Attempt to pick each object
        for obj_name in object_names:
            obj_pos = positions[obj_name]
            print(f"[Exploration] Attempting to pick {obj_name} at {obj_pos}")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after picking {obj_name}!")
                    break
            except Exception as e:
                print(f"[Exploration] Exception during pick of {obj_name}: {e}")
                missing_predicates.add('at or handempty or holding')

        # Attempt to place each object (if possible)
        for obj_name in object_names:
            obj_pos = positions[obj_name]
            print(f"[Exploration] Attempting to place {obj_name} at {obj_pos}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after placing {obj_name}!")
                    break
            except Exception as e:
                print(f"[Exploration] Exception during place of {obj_name}: {e}")
                missing_predicates.add('at or handempty or holding')

        # Attempt to rotate and pull (if possible)
        # For rotate and pull, we need to know gripper, drawer, angle, etc.
        # We'll try to infer these from the positions dictionary if possible.
        # This is a generic exploration; in a real scenario, you would use actual names/types.

        # Try to find a gripper, drawer, and angle in the positions
        gripper_name = None
        drawer_name = None
        angle_from = None
        angle_to = None

        for name in object_names:
            if 'gripper' in name:
                gripper_name = name
            if 'drawer' in name:
                drawer_name = name
            if 'angle' in name and angle_from is None:
                angle_from = name
            elif 'angle' in name and angle_to is None:
                angle_to = name

        # Try rotate if possible
        if gripper_name and angle_from and angle_to:
            print(f"[Exploration] Attempting to rotate {gripper_name} from {angle_from} to {angle_to}")
            try:
                obs, reward, done = rotate(
                    env,
                    task,
                    gripper=gripper_name,
                    from_angle=angle_from,
                    to_angle=angle_to
                )
                if done:
                    print(f"[Exploration] Task ended after rotate!")
            except Exception as e:
                print(f"[Exploration] Exception during rotate: {e}")
                missing_predicates.add('rotated or angle')

        # Try pull if possible
        if gripper_name and drawer_name:
            print(f"[Exploration] Attempting to pull {drawer_name} with {gripper_name}")
            try:
                obs, reward, done = pull(
                    env,
                    task,
                    gripper=gripper_name,
                    drawer=drawer_name
                )
                if done:
                    print(f"[Exploration] Task ended after pull!")
            except Exception as e:
                print(f"[Exploration] Exception during pull: {e}")
                missing_predicates.add('holding-drawer or is-locked or is-open')

        # === Exploration Summary ===
        if missing_predicates:
            print("[Exploration] Potential missing predicates detected:")
            for pred in missing_predicates:
                print(f"    - {pred}")
        else:
            print("[Exploration] No missing predicates detected during skill execution.")

        # === End of Exploration Phase ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()