# run_skeleton_task.py (Completed with Exploration Phase and Feedback Fix)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

import time

# === FEEDBACK FIX: Add euler_from_quat utility ===
def euler_from_quat(quaternion):
    # Convert quaternion (x, y, z, w) to Euler angles (roll, pitch, yaw)
    x, y, z, w = quaternion
    # roll (x-axis rotation)
    t0 = +2.0 * (w * x + y * z)
    t1 = +1.0 - 2.0 * (x * x + y * y)
    roll_x = np.arctan2(t0, t1)
    # pitch (y-axis rotation)
    t2 = +2.0 * (w * y - z * x)
    t2 = np.clip(t2, -1.0, 1.0)
    pitch_y = np.arcsin(t2)
    # yaw (z-axis rotation)
    t3 = +2.0 * (w * z + x * y)
    t4 = +1.0 - 2.0 * (y * y + z * z)
    yaw_z = np.arctan2(t3, t4)
    return roll_x, pitch_y, yaw_z

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        print("[Exploration] Object positions:", positions)

        # === EXPLORATION PHASE: Try to discover missing predicates ===
        # The goal is to determine which predicate is missing by performing available actions and observing the effects.
        # We will attempt to use each available skill and print out any new information or errors.

        available_skills = ['pick', 'place', 'move', 'rotate', 'pull']
        exploration_results = {}

        # Try to pick up each object at its location (if possible)
        for obj_name, obj_info in positions.items():
            obj_pos = obj_info.get('position', None)
            obj_loc = obj_info.get('location', None)
            if obj_pos is None and obj_loc is None:
                continue
            print(f"[Exploration] Attempting to pick object: {obj_name}")
            try:
                # Try pick (if skill signature is pick(env, task, target_pos, ...))
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos if obj_pos is not None else [0,0,0],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                exploration_results[f'pick_{obj_name}'] = (obs, reward, done)
                print(f"[Exploration] Picked {obj_name}: reward={reward}, done={done}")
                if done:
                    print("[Exploration] Task ended after pick!")
                    break
            except Exception as e:
                print(f"[Exploration] Pick failed for {obj_name}: {e}")

        # Try to place (if holding something)
        for obj_name, result in exploration_results.items():
            obs, reward, done = result
            if 'pick_' in obj_name and not done:
                # Try to place at a random location (if possible)
                for loc_name, loc_info in positions.items():
                    if loc_name == obj_name.replace('pick_', ''):
                        continue
                    loc_pos = loc_info.get('position', None)
                    if loc_pos is None:
                        continue
                    print(f"[Exploration] Attempting to place {obj_name.replace('pick_', '')} at {loc_name}")
                    try:
                        obs, reward, done = place(
                            env,
                            task,
                            target_pos=loc_pos,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        print(f"[Exploration] Placed {obj_name.replace('pick_', '')} at {loc_name}: reward={reward}, done={done}")
                        if done:
                            print("[Exploration] Task ended after place!")
                            break
                    except Exception as e:
                        print(f"[Exploration] Place failed for {obj_name.replace('pick_', '')} at {loc_name}: {e}")

        # Try to move (if move skill is available and signature is move(env, task, from_pos, to_pos, ...))
        # We'll attempt to move between two locations if possible
        loc_names = list(positions.keys())
        if len(loc_names) >= 2:
            from_loc = positions[loc_names[0]].get('position', None)
            to_loc = positions[loc_names[1]].get('position', None)
            if from_loc is not None and to_loc is not None:
                print(f"[Exploration] Attempting to move from {loc_names[0]} to {loc_names[1]}")
                try:
                    obs, reward, done = move(
                        env,
                        task,
                        from_pos=from_loc,
                        to_pos=to_loc,
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                    print(f"[Exploration] Moved from {loc_names[0]} to {loc_names[1]}: reward={reward}, done={done}")
                except Exception as e:
                    print(f"[Exploration] Move failed: {e}")

        # Try to rotate (if rotate skill is available)
        # We'll attempt to rotate the gripper to a known orientation (e.g., identity quaternion)
        try:
            print("[Exploration] Attempting to rotate gripper to identity quaternion")
            identity_quat = np.array([0, 0, 0, 1])
            obs, reward, done = rotate(
                env,
                task,
                target_quat=identity_quat,
                max_steps=50,
                threshold=0.05,
                timeout=5.0
            )
            print(f"[Exploration] Rotated gripper: reward={reward}, done={done}")
        except Exception as e:
            print(f"[Exploration] Rotate failed: {e}")

        # Try to pull (if pull skill is available and there is a drawer or pullable object)
        for obj_name, obj_info in positions.items():
            if 'drawer' in obj_name or 'handle' in obj_name:
                print(f"[Exploration] Attempting to pull {obj_name}")
                try:
                    obs, reward, done = pull(
                        env,
                        task,
                        target_name=obj_name,
                        max_steps=50,
                        threshold=0.01,
                        timeout=5.0
                    )
                    print(f"[Exploration] Pulled {obj_name}: reward={reward}, done={done}")
                except Exception as e:
                    print(f"[Exploration] Pull failed for {obj_name}: {e}")

        print("[Exploration] Exploration phase complete. Check logs for missing predicate clues.")

        # === END OF EXPLORATION PHASE ===

        # === (Optional) Main Task Plan ===
        # After exploration, you would implement the oracle plan here using only the available skills.
        # For this code, the focus is on exploration for missing predicates as per the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
