# run_skeleton_task.py (Completed with Exploration Phase and Predicate Discovery)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Discover Missing Predicate ===
        # The goal is to explore the environment to discover which predicate is missing,
        # as per the feedback and exploration knowledge.

        # 1. Identify all objects and their locations
        print("[Exploration] Identifying objects and their locations...")
        identified_objects = []
        for obj_name, obj_info in positions.items():
            # obj_info may contain 'type', 'position', 'location', etc.
            # For each object, try to move to its location to 'identify' it
            try:
                obj_loc = obj_info.get('location', None)
                if obj_loc is None:
                    print(f"[Exploration] Warning: No location info for {obj_name}, skipping.")
                    continue
                # Use 'move' skill to approach the object's location
                print(f"[Exploration] Moving to location of {obj_name}: {obj_loc}")
                # The move skill may require (env, task, from_pos, to_pos, ...)
                # For simplicity, assume we can move directly to obj_loc
                obs, reward, done = move(env, task, target_pos=obj_info['position'])
                identified_objects.append(obj_name)
                if done:
                    print(f"[Exploration] Task ended during move to {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during move to {obj_name}: {e}")

        # 2. For each object, try to pick it up to discover weight/durability
        print("[Exploration] Attempting to pick objects to discover weight/durability...")
        for obj_name in identified_objects:
            obj_info = positions[obj_name]
            try:
                obj_pos = obj_info['position']
                print(f"[Exploration] Attempting to pick {obj_name} at {obj_pos}")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after picking {obj_name}!")
                    return
                # Optionally, place the object back
                print(f"[Exploration] Placing {obj_name} back at {obj_pos}")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after placing {obj_name}!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during pick/place of {obj_name}: {e}")

        # 3. For drawers or objects with handles, try to pull to discover lock status
        print("[Exploration] Attempting to pull drawers to discover lock status...")
        for obj_name, obj_info in positions.items():
            if obj_info.get('type', '') == 'drawer':
                try:
                    drawer_pos = obj_info['position']
                    print(f"[Exploration] Moving to drawer {obj_name} at {drawer_pos}")
                    obs, reward, done = move(env, task, target_pos=drawer_pos)
                    if done:
                        print(f"[Exploration] Task ended during move to drawer {obj_name}!")
                        return
                    print(f"[Exploration] Attempting to pull {obj_name}")
                    obs, reward, done = pull(env, task, target_pos=drawer_pos)
                    if done:
                        print(f"[Exploration] Task ended after pulling {obj_name}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Exception during pull of {obj_name}: {e}")

        # 4. For objects that require rotation (e.g., handles), try to rotate
        print("[Exploration] Attempting to rotate objects (if applicable)...")
        for obj_name, obj_info in positions.items():
            if obj_info.get('can_rotate', False):
                try:
                    obj_pos = obj_info['position']
                    # Define a target quaternion for rotation (example: 90 degrees about z)
                    target_quat = [0, 0, np.sin(np.pi/4), np.cos(np.pi/4)]
                    print(f"[Exploration] Rotating {obj_name} to target quaternion {target_quat}")
                    obs, reward, done = rotate(
                        env,
                        task,
                        target_quat=target_quat,
                        max_steps=100,
                        threshold=0.05,
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after rotating {obj_name}!")
                        return
                except Exception as e:
                    print(f"[Exploration] Exception during rotate of {obj_name}: {e}")

        # 5. Special handling for trash object (if present)
        print("[Exploration] Checking for trash object handling...")
        for obj_name, obj_info in positions.items():
            if 'trash' in obj_name.lower():
                try:
                    trash_pos = obj_info['position']
                    print(f"[Exploration] Attempting to pick trash object {obj_name} at {trash_pos}")
                    obs, reward, done = pick(
                        env,
                        task,
                        target_pos=trash_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Exploration] Task ended after picking trash {obj_name}!")
                        return
                    # Optionally, move to disposal location and place
                    disposal_pos = positions.get('disposal_bin', {}).get('position', None)
                    if disposal_pos is not None:
                        print(f"[Exploration] Placing trash {obj_name} in disposal bin at {disposal_pos}")
                        obs, reward, done = move(env, task, target_pos=disposal_pos)
                        if done:
                            print(f"[Exploration] Task ended during move to disposal bin!")
                            return
                        obs, reward, done = place(
                            env,
                            task,
                            target_pos=disposal_pos,
                            approach_distance=0.15,
                            max_steps=100,
                            threshold=0.01,
                            approach_axis='z',
                            timeout=10.0
                        )
                        if done:
                            print(f"[Exploration] Task ended after placing trash {obj_name}!")
                            return
                except Exception as e:
                    print(f"[Exploration] Exception during trash handling for {obj_name}: {e}")

        print("[Exploration] Exploration phase complete. If any predicate was missing, it should be discoverable from the above actions.")

        # === Main Task Plan (if any) ===
        # At this point, you can proceed to execute the oracle plan using the available skills,
        # now that the environment and predicates have been explored and validated.
        # (Insert oracle plan execution here if provided.)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
