# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer1': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Check Drawer Lock State ===
        # Feedback: (not (is-locked drawer1))
        # We need to determine if drawer1 is locked or unlocked before attempting to pull.
        # The only available skills are: ['pick', 'place', 'move', 'rotate', 'pull']
        # According to exploration knowledge, the 'pull' action can be used to discover the lock state.

        # For this example, we assume:
        # - The gripper must be at the anchor position of drawer1 to pull.
        # - The gripper must be rotated to the correct angle (e.g., 'ninety_deg') before moving to the anchor.
        # - The plan is: rotate gripper -> move to side pos -> move to anchor pos -> pick-drawer (grasp handle) -> pull (explore lock state) -> pull (open drawer if unlocked).

        # --- Define object and position names (replace with actual names from your environment) ---
        drawer_name = 'drawer1'
        gripper_name = 'gripper1'
        # These should be keys in positions, or you may need to adjust based on your environment
        # For demonstration, we use placeholder names:
        side_pos_name = 'drawer1_side'
        anchor_pos_name = 'drawer1_anchor'
        angle_from = 'zero_deg'
        angle_to = 'ninety_deg'

        # --- Step 1: Rotate gripper to 90 degrees (if not already) ---
        try:
            print(f"[Exploration] Rotating {gripper_name} from {angle_from} to {angle_to}")
            obs, reward, done = rotate(env, task, gripper_name, angle_from, angle_to)
            if done:
                print("[Exploration] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during rotate: {e}")

        # --- Step 2: Move gripper to side position of drawer ---
        try:
            print(f"[Exploration] Moving {gripper_name} to side position {side_pos_name} of {drawer_name}")
            obs, reward, done = move(env, task, gripper_name, drawer_name, None, side_pos_name)
            if done:
                print("[Exploration] Task ended during move-to-side!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during move-to-side: {e}")

        # --- Step 3: Move gripper to anchor position of drawer ---
        try:
            print(f"[Exploration] Moving {gripper_name} from side {side_pos_name} to anchor {anchor_pos_name} of {drawer_name}")
            obs, reward, done = move(env, task, gripper_name, drawer_name, side_pos_name, anchor_pos_name)
            if done:
                print("[Exploration] Task ended during move-to-anchor!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during move-to-anchor: {e}")

        # --- Step 4: Pick (grasp) the drawer handle at anchor position ---
        try:
            print(f"[Exploration] Picking (grasping) {drawer_name} with {gripper_name} at {anchor_pos_name}")
            obs, reward, done = pick(env, task, gripper_name, drawer_name, anchor_pos_name)
            if done:
                print("[Exploration] Task ended during pick-drawer!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pick-drawer: {e}")

        # --- Step 5: Exploration Pull to Discover Lock State ---
        try:
            print(f"[Exploration] Pulling {drawer_name} to check lock state")
            # Use a small pull distance for exploration
            pull_distance = 0.01  # Small value to test lock
            obs, reward, done = pull(env, task, pull_distance, pull_axis='x', max_steps=30, threshold=0.005, timeout=3.0)
            if done:
                print("[Exploration] Task ended during exploration pull!")
                return
            # After this, the robot should know if the drawer is locked or not (predicate discovered)
        except Exception as e:
            print(f"[Exploration] Exception during exploration pull: {e}")

        # --- Step 6: If drawer is unlocked, perform full pull to open ---
        # In a real system, you would check the observation or state to see if the drawer is openable.
        # For this skeleton, we proceed to attempt a full pull.
        try:
            print(f"[Task] Attempting to fully pull open {drawer_name}")
            full_pull_distance = 0.15  # Example value for full open
            obs, reward, done = pull(env, task, full_pull_distance, pull_axis='x', max_steps=100, threshold=0.01, timeout=10.0)
            if done:
                print("[Task] Task ended after full pull!")
                return
        except Exception as e:
            print(f"[Task] Exception during full pull: {e}")

        # --- Task Complete ---
        print("[Task] Drawer manipulation and exploration complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
