# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'item3': (x, y, z), 'table': (x, y, z), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (at item3 table) is missing
        # We need to check if the robot can detect the presence of item3 at the table
        # and if the predicate (at item3 table) is established after exploration.

        # For this, we will attempt to move to the table and check for item3
        # This simulates the exploration to discover the missing predicate.

        # Defensive: Check if 'item3' and 'table' exist in positions
        if 'item3' not in positions or 'table' not in positions:
            print("[Exploration] Error: 'item3' or 'table' not found in object positions.")
            return

        item3_pos = positions['item3']
        table_pos = positions['table']

        print("[Exploration] Moving to table to identify objects present...")

        # Use the 'move' skill to approach the table (if available)
        # Since the move skill signature is not provided, we assume it moves the gripper to a position
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=table_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Exploration] Exception during move to table: {e}")
            return

        # After moving, check if item3 is at the table (simulate perception)
        # In a real system, this would involve perception; here, we use the positions dict
        dist = np.linalg.norm(np.array(item3_pos) - np.array(table_pos))
        if dist < 0.2:
            print("[Exploration] item3 is at the table. (at item3 table) predicate is TRUE.")
        else:
            print("[Exploration] item3 is NOT at the table. (at item3 table) predicate is FALSE or missing.")

        # === Task Plan Execution ===
        # Now, execute the plan to pick item3 from the table

        print("[Task] Attempting to pick item3 from the table...")

        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=item3_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Exception during pick: {e}")
            return

        if done:
            print("[Task] Task ended after picking item3!")
            return

        # Optionally, place item3 at another location if required by the plan
        # For demonstration, let's place it back on the table (or another location if specified)
        print("[Task] Placing item3 back on the table...")

        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=table_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        except Exception as e:
            print(f"[Task] Exception during place: {e}")
            return

        if done:
            print("[Task] Task ended after placing item3!")
            return

        print("[Task] item3 pick and place completed.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()