# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'bin': ..., 'bottom': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Check for Missing Predicate (is-side-pos bin bottom) ===
        # The feedback indicates that the predicate (is-side-pos bin bottom) is missing.
        # We will attempt to explore the environment to check if 'bin' and 'bottom' exist and their relation.

        print("=== [Exploration] Checking for missing predicate: (is-side-pos bin bottom) ===")
        bin_pos = None
        bottom_pos = None

        # Try to retrieve 'bin' and 'bottom' positions from the environment
        try:
            bin_pos = positions.get('bin', None)
            bottom_pos = positions.get('bottom', None)
        except Exception as e:
            print("[Exploration] Error retrieving object positions:", e)

        if bin_pos is not None and bottom_pos is not None:
            print(f"[Exploration] Found 'bin' at {bin_pos}, 'bottom' at {bottom_pos}.")
            # Try to move to 'bottom' and see if it is a side position of 'bin'
            try:
                print("[Exploration] Moving to 'bottom' position to check relation with 'bin'.")
                obs, reward, done = move(env, task, target_pos=bottom_pos)
                # After moving, we could check for any observable effect or state change
                # Since we cannot directly check predicates, we log the exploration
                print("[Exploration] Arrived at 'bottom'. If this is a side position of 'bin', the predicate is valid.")
            except Exception as e:
                print("[Exploration] Error during move to 'bottom':", e)
        else:
            print("[Exploration] Could not find both 'bin' and 'bottom' in object positions. Predicate remains unverified.")

        # === Main Task Plan (Example) ===
        # Here you would execute the oracle plan using only the available skills.
        # For demonstration, we show how to use the available skills in sequence.
        # Replace the following with the actual plan steps as needed.

        # Example: Pick an object from 'bin' and place it at 'bottom' if both exist
        if bin_pos is not None and bottom_pos is not None:
            # Assume there is an object to pick at 'bin'
            # For demonstration, we try to pick and place a generic object if present
            object_to_pick = None
            for obj_name, obj_pos in positions.items():
                if obj_name not in ['bin', 'bottom']:
                    # Check if the object is close to 'bin'
                    if np.linalg.norm(np.array(obj_pos) - np.array(bin_pos)) < 0.05:
                        object_to_pick = obj_name
                        break
            if object_to_pick is not None:
                print(f"[Task] Attempting to pick '{object_to_pick}' from 'bin' and place at 'bottom'.")
                try:
                    # Move to bin
                    obs, reward, done = move(env, task, target_pos=bin_pos)
                    if done:
                        print("[Task] Task ended during move to bin!")
                        return
                    # Pick the object
                    obs, reward, done = pick(env, task, target_pos=bin_pos)
                    if done:
                        print("[Task] Task ended during pick!")
                        return
                    # Move to bottom
                    obs, reward, done = move(env, task, target_pos=bottom_pos)
                    if done:
                        print("[Task] Task ended during move to bottom!")
                        return
                    # Place the object
                    obs, reward, done = place(env, task, target_pos=bottom_pos)
                    if done:
                        print("[Task] Task ended after place!")
                        return
                    print("[Task] Successfully picked and placed the object.")
                except Exception as e:
                    print("[Task] Error during pick and place sequence:", e)
            else:
                print("[Task] No suitable object found near 'bin' to pick.")
        else:
            print("[Task] Skipping pick and place: 'bin' or 'bottom' not found.")

        # === End of Task Plan ===

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()