# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'gripper': ..., 'bin': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (gripper-at gripper bin)
        # The predicate (gripper-at gripper bin) is missing or needs to be verified.
        # We will attempt to move the gripper to the 'bin' position and check if the predicate holds.

        gripper_name = 'gripper'
        bin_name = 'bin'

        # Defensive: Check if required objects exist in positions
        if gripper_name not in positions:
            print(f"[Exploration] ERROR: '{gripper_name}' not found in object positions.")
            return
        if bin_name not in positions:
            print(f"[Exploration] ERROR: '{bin_name}' not found in object positions.")
            return

        gripper_pos = positions[gripper_name]
        bin_pos = positions[bin_name]

        print(f"[Exploration] Moving gripper to bin position: {bin_pos}")

        # Use the provided 'move' skill to move the gripper to the bin
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(bin_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        except Exception as e:
            print(f"[Exploration] Exception during move: {e}")
            return

        # After moving, check if the predicate (gripper-at gripper bin) is now true
        # (In a real system, this would be checked via the environment's state or observation)
        # For demonstration, we print the current gripper position
        obs = task.get_observation()
        current_gripper_pos = obs.gripper_pose[:3]
        print(f"[Exploration] Gripper current position: {current_gripper_pos}")
        print(f"[Exploration] Target bin position: {bin_pos}")
        dist = np.linalg.norm(np.array(current_gripper_pos) - np.array(bin_pos))
        if dist < 0.02:
            print("[Exploration] Predicate (gripper-at gripper bin) is now TRUE (within threshold).")
        else:
            print("[Exploration] Predicate (gripper-at gripper bin) is FALSE (not at bin).")

        # === End of Exploration Phase ===

        # === (Optional) Continue with Oracle Plan or Task Steps ===
        # Here you would continue with the main plan, e.g., pick, place, rotate, pull, etc.
        # For this exploration, we focus on the missing predicate as per feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()