# run_skeleton_task.py (Completed with Exploration Phase)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (holding anchor-pos-bottom)
        # The feedback suggests that the predicate (holding anchor-pos-bottom) is missing.
        # We need to explore and check if the robot can pick or interact with 'anchor-pos-bottom'.
        # We'll attempt to pick 'anchor-pos-bottom' and observe the result.

        anchor_pos_name = 'anchor-pos-bottom'
        if anchor_pos_name not in positions:
            print(f"[Exploration] '{anchor_pos_name}' not found in object positions. Exploration aborted.")
        else:
            anchor_pos = positions[anchor_pos_name]
            print(f"[Exploration] Attempting to pick '{anchor_pos_name}' at position {anchor_pos}.")

            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=anchor_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after picking '{anchor_pos_name}'.")
                    return
                else:
                    print(f"[Exploration] Successfully executed pick on '{anchor_pos_name}'.")
            except Exception as e:
                print(f"[Exploration] Exception during pick: {e}")

        # === Main Task Plan (Oracle Plan Execution) ===
        # After exploration, proceed with the main plan using available skills.
        # Example: If the goal is to pick and place an object, do so here.

        # For demonstration, let's assume the next step is to place the anchor at a target location if required.
        # We'll check for a plausible 'target_location' in positions.
        target_location_name = 'target-location'
        if target_location_name in positions:
            target_location = positions[target_location_name]
            print(f"[Task] Attempting to place '{anchor_pos_name}' at '{target_location_name}' ({target_location}).")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=target_location,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Task] Task ended after placing '{anchor_pos_name}'.")
                    return
                else:
                    print(f"[Task] Successfully executed place for '{anchor_pos_name}'.")
            except Exception as e:
                print(f"[Task] Exception during place: {e}")
        else:
            print(f"[Task] No 'target-location' found in positions. Skipping place step.")

        # If there are additional steps in the oracle plan, add them here using only the available skills:
        # ['pick', 'place', 'move', 'rotate', 'pull']
        # For example, you could add:
        # - move(env, task, from_pos, to_pos)
        # - rotate(env, task, ...)
        # - pull(env, task, ...)

        # (Add further plan steps as needed, using only the provided skills.)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
