# run_skeleton_task.py (Completed with Exploration Phase for Missing Predicate)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example: positions = {'drawer1': {...}, 'posC': np.array([...]), ...}
        positions = get_object_positions()

        # === Exploration Phase: Check for Missing Predicate (is-side-pos posC drawer1) ===
        # Feedback indicates (is-side-pos posC drawer1) is missing or needs to be explored.
        # We will attempt to move the gripper to posC near drawer1 and observe the result.

        # Defensive: Check if required keys exist
        if 'drawer1' not in positions or 'posC' not in positions:
            print("[Exploration] Required objects/positions ('drawer1', 'posC') not found in positions dictionary.")
            print("[Exploration] Available keys:", list(positions.keys()))
            return

        drawer1_pos = positions['drawer1']
        posC = positions['posC']

        print("[Exploration] Moving to posC near drawer1 to explore (is-side-pos posC drawer1)")

        # Move the gripper to posC (side position of drawer1)
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=posC,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended during exploration move to posC.")
                return
        except Exception as e:
            print(f"[Exploration] Exception during move to posC: {e}")
            return

        # At this point, the robot is at posC. In a real exploration, we might check sensors or state.
        # For this skeleton, we simply log that the exploration step is complete.
        print("[Exploration] Exploration move to posC complete. (Predicate (is-side-pos posC drawer1) can now be considered known.)")

        # === Main Task Plan (Example) ===
        # After exploration, you would proceed with the main plan using the available skills.
        # For demonstration, we show a generic sequence using the available skills.
        # Replace the following with your actual oracle plan as needed.

        # Example: Pick an object at posC (if any), then place it elsewhere.
        # (This is a placeholder; adapt to your actual task.)

        # Let's check if there is an object at posC to pick
        object_to_pick = None
        for obj_name, obj_info in positions.items():
            # Assume obj_info contains 'position' and 'type' or similar
            if isinstance(obj_info, dict) and 'position' in obj_info:
                obj_pos = obj_info['position']
            elif isinstance(obj_info, np.ndarray):
                obj_pos = obj_info
            else:
                continue
            if np.allclose(obj_pos, posC, atol=0.02) and obj_name != 'drawer1':
                object_to_pick = obj_name
                break

        if object_to_pick:
            print(f"[Task] Attempting to pick object '{object_to_pick}' at posC.")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=posC,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after pick.")
                    return
            except Exception as e:
                print(f"[Task] Exception during pick: {e}")
        else:
            print("[Task] No object found at posC to pick.")

        # Example: Rotate the gripper if needed (demonstration)
        # (Assume rotate takes env, task, gripper_name, from_angle, to_angle)
        # If your plan requires, you can call rotate here.

        # Example: Pull the drawer if needed (demonstration)
        # (Assume pull takes env, task, drawer_name)
        # If your plan requires, you can call pull here.

        # You can continue with further plan steps as required by your oracle plan.

        print("[Task] Skeleton task completed main plan steps.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()