# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback and exploration knowledge indicate we need to check for missing predicates
        # such as 'weight-known', 'durability-known', 'identified', 'temperature-known', etc.
        # We will perform exploration actions to discover these properties.

        # Step 1: List all objects and their positions
        print("[Exploration] Listing all objects and their positions:")
        for obj_name, obj_pos in positions.items():
            print(f"    {obj_name}: {obj_pos}")

        # Step 2: For each object, move to its location and attempt to pick it up
        #         This will help us discover if the object is pickable and if we can learn its properties
        for obj_name, obj_pos in positions.items():
            print(f"[Exploration] Attempting to move to and pick up: {obj_name}")

            # Move to the object's position
            try:
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Exploration] Failed to move to {obj_name}: {e}")
                continue

            # Try to pick the object (if pickable)
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Picked up {obj_name}.")
            except Exception as e:
                print(f"[Exploration] Could not pick {obj_name}: {e}")
                continue

            # Try to place the object back (if possible)
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Exploration] Placed {obj_name} back.")
            except Exception as e:
                print(f"[Exploration] Could not place {obj_name}: {e}")

            if done:
                print("[Exploration] Task ended during exploration!")
                return

        # Step 3: Attempt to rotate and pull drawers if present
        #         This helps discover if objects are drawers and if they are locked/open
        for obj_name, obj_pos in positions.items():
            if "drawer" in obj_name:
                print(f"[Exploration] Attempting to rotate and pull drawer: {obj_name}")
                # Try rotating the gripper to 90 degrees (if required)
                try:
                    obs, reward, done = rotate(
                        env,
                        task,
                        gripper_name="gripper",  # Assuming single gripper named 'gripper'
                        from_angle="zero_deg",
                        to_angle="ninety_deg"
                    )
                    print(f"[Exploration] Rotated gripper for {obj_name}.")
                except Exception as e:
                    print(f"[Exploration] Could not rotate for {obj_name}: {e}")

                # Try pulling the drawer
                try:
                    obs, reward, done = pull(
                        env,
                        task,
                        drawer_name=obj_name
                    )
                    print(f"[Exploration] Pulled {obj_name}.")
                except Exception as e:
                    print(f"[Exploration] Could not pull {obj_name}: {e}")

                if done:
                    print("[Exploration] Task ended during drawer exploration!")
                    return

        # Step 4: Safety Constraints and Force Calibration
        #         (Feedback: Ensure we do not throw away fragile/valuable objects)
        #         Here, we would check object properties (if available) and avoid unsafe actions.
        #         Since we do not have direct access to object types, we skip unsafe actions.

        print("[Exploration] Exploration phase complete. If any missing predicate was discovered, update your plan accordingly.")

        # === Main Task Plan (Oracle Plan Execution) ===
        # After exploration, proceed with the main plan using only available skills.
        # For demonstration, we show a generic pick-and-place for all objects (if safe).

        for obj_name, obj_pos in positions.items():
            print(f"[Task] Executing main plan for: {obj_name}")

            # Move to object
            try:
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Task] Failed to move to {obj_name}: {e}")
                continue

            # Pick object
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=np.array(obj_pos),
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                print(f"[Task] Picked up {obj_name}.")
            except Exception as e:
                print(f"[Task] Could not pick {obj_name}: {e}")
                continue

            # Decide where to place the object (e.g., a bin or designated location)
            # For demonstration, if 'bin' exists in positions, place object there
            if 'bin' in positions:
                bin_pos = positions['bin']
                try:
                    obs, reward, done = move(
                        env,
                        task,
                        target_pos=np.array(bin_pos),
                        max_steps=100,
                        threshold=0.01,
                        timeout=10.0
                    )
                except Exception as e:
                    print(f"[Task] Failed to move to bin for {obj_name}: {e}")
                    continue

                try:
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=np.array(bin_pos),
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    print(f"[Task] Placed {obj_name} in bin.")
                except Exception as e:
                    print(f"[Task] Could not place {obj_name} in bin: {e}")
            else:
                print(f"[Task] No bin found for placing {obj_name}. Skipping place step.")

            if done:
                print("[Task] Task ended after main plan execution!")
                return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()