# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

import time

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'object_1': (0,0,0), 'object_2': (1,1,1), ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The feedback indicates we need to ensure the gripper is at the target position before holding,
        # and to hold for a duration to ensure stability.
        # We'll perform an exploration step to check if the predicate (gripper-at gripper bin) is satisfied
        # before attempting to pick/place.

        # For demonstration, let's assume we want to pick up 'object_1' and place it at 'bin'
        # and that the gripper must be at the 'bin' position before placing.

        # Replace these with actual object/location names from your environment
        object_name = 'object_1'
        bin_name = 'bin'
        gripper_name = 'gripper'

        # Get positions for the object and bin
        try:
            object_pos = positions[object_name]
            bin_pos = positions[bin_name]
        except KeyError as e:
            print(f"[Error] Missing object in positions: {e}")
            shutdown_environment(env)
            return

        # === Step 1: Move gripper to object position ===
        print(f"[Task] Moving gripper to {object_name} at position {object_pos}")
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(object_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during move to object!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to object: {e}")
            shutdown_environment(env)
            return

        # === Step 2: Ensure gripper is at object position (predicate check) ===
        gripper_at_object = False
        obs = task.get_observation()
        gripper_pos = obs.gripper_pose[:3]
        if np.linalg.norm(gripper_pos - np.array(object_pos)) < 0.01:
            gripper_at_object = True
        else:
            print("[Warning] Gripper not at object position after move. Attempting to correct.")
            # Try to move again
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(object_pos),
                max_steps=50,
                threshold=0.01,
                timeout=5.0
            )
            obs = task.get_observation()
            gripper_pos = obs.gripper_pose[:3]
            if np.linalg.norm(gripper_pos - np.array(object_pos)) < 0.01:
                gripper_at_object = True

        if not gripper_at_object:
            print("[Error] Gripper could not reach object position. Aborting.")
            shutdown_environment(env)
            return

        # Hold at object position for stability (hold duration)
        print("[Task] Holding gripper at object position for stability.")
        hold_steps = 5
        for _ in range(hold_steps):
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(object_pos),
                max_steps=1,
                threshold=0.01,
                timeout=1.0
            )
            if done:
                print("[Task] Task ended during hold at object!")
                return
            time.sleep(0.05)

        # === Step 3: Pick the object ===
        print(f"[Task] Picking up {object_name}")
        try:
            obs, reward, done = pick(
                env,
                task,
                target_pos=np.array(object_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during pick!")
                return
        except Exception as e:
            print(f"[Error] Exception during pick: {e}")
            shutdown_environment(env)
            return

        # === Step 4: Move gripper to bin position ===
        print(f"[Task] Moving gripper to {bin_name} at position {bin_pos}")
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(bin_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during move to bin!")
                return
        except Exception as e:
            print(f"[Error] Exception during move to bin: {e}")
            shutdown_environment(env)
            return

        # === Step 5: Ensure gripper is at bin position (predicate check) ===
        gripper_at_bin = False
        obs = task.get_observation()
        gripper_pos = obs.gripper_pose[:3]
        if np.linalg.norm(gripper_pos - np.array(bin_pos)) < 0.01:
            gripper_at_bin = True
        else:
            print("[Warning] Gripper not at bin position after move. Attempting to correct.")
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(bin_pos),
                max_steps=50,
                threshold=0.01,
                timeout=5.0
            )
            obs = task.get_observation()
            gripper_pos = obs.gripper_pose[:3]
            if np.linalg.norm(gripper_pos - np.array(bin_pos)) < 0.01:
                gripper_at_bin = True

        if not gripper_at_bin:
            print("[Error] Gripper could not reach bin position. Aborting.")
            shutdown_environment(env)
            return

        # Hold at bin position for stability (hold duration)
        print("[Task] Holding gripper at bin position for stability.")
        hold_steps = 5
        for _ in range(hold_steps):
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(bin_pos),
                max_steps=1,
                threshold=0.01,
                timeout=1.0
            )
            if done:
                print("[Task] Task ended during hold at bin!")
                return
            time.sleep(0.05)

        # === Step 6: Place the object ===
        print(f"[Task] Placing {object_name} at {bin_name}")
        try:
            obs, reward, done = place(
                env,
                task,
                target_pos=np.array(bin_pos),
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Task] Task ended during place!")
                return
        except Exception as e:
            print(f"[Error] Exception during place: {e}")
            shutdown_environment(env)
            return

        print("[Task] Successfully completed pick and place with stability checks.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()