# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'gripper': ..., 'bin': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (gripper-at gripper bin) is missing in the initial state.
        # We need to ensure the gripper is at the 'bin' position before any manipulation.
        # The available skills are: ['pick', 'place', 'move', 'rotate', 'pull']
        # So, we use 'move' to bring the gripper to the 'bin' position.

        # Defensive: Check that required objects exist in positions
        if 'gripper' not in positions:
            print("[Error] 'gripper' position not found in object positions.")
            return
        if 'bin' not in positions:
            print("[Error] 'bin' position not found in object positions.")
            return

        gripper_pos = positions['gripper']
        bin_pos = positions['bin']

        print(f"[Exploration] Moving gripper from its current position to 'bin' position to establish (gripper-at gripper bin).")
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=bin_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended during move to bin!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during move: {e}")
            return

        # After this move, the predicate (gripper-at gripper bin) should be true in the environment.
        print("[Exploration] (gripper-at gripper bin) predicate established.")

        # === Main Task Plan (Example) ===
        # At this point, you would continue with the oracle plan, e.g., pick, place, rotate, pull, etc.
        # For demonstration, let's try to pick an object at the bin (if any), using only available skills.

        # Find an object at the bin (other than the gripper itself)
        object_to_pick = None
        for obj_name, obj_pos in positions.items():
            if obj_name not in ['gripper', 'bin']:
                # If the object's position is close to the bin, consider it at the bin
                if np.linalg.norm(np.array(obj_pos) - np.array(bin_pos)) < 0.05:
                    object_to_pick = obj_name
                    break

        if object_to_pick is not None:
            print(f"[Task] Attempting to pick object '{object_to_pick}' at bin.")
            try:
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=bin_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after pick!")
                    return
            except Exception as e:
                print(f"[Task] Exception during pick: {e}")
                return
        else:
            print("[Task] No pickable object found at bin.")

        # You can continue with further actions (place, rotate, pull, etc.) as per your oracle plan.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()