# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer0': ..., 'pos0': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Check for Missing Predicate (is-side-pos) ===
        # Feedback indicates (is-side-pos pos0 drawer0) is missing or needs to be checked.
        # We'll attempt to move the gripper to pos0 and interact with drawer0 to explore this.
        # This phase is to help discover if the predicate is needed for the plan.

        # Defensive: Check if required keys exist
        drawer_name = 'drawer0'
        pos_name = 'pos0'
        if drawer_name not in positions or pos_name not in positions:
            print(f"[Exploration] Required objects '{drawer_name}' or '{pos_name}' not found in positions dictionary.")
            return

        drawer_pos = positions[drawer_name]
        side_pos = positions[pos_name]

        print(f"[Exploration] Moving gripper to side position '{pos_name}' for drawer '{drawer_name}' at {side_pos}.")

        # 1. Move to side position (pos0) to check if (is-side-pos pos0 drawer0) is satisfied
        try:
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(side_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended during move to side position!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during move to side position: {e}")
            return

        # 2. Try to rotate the gripper to the required angle for the next action (if needed)
        #    For demonstration, let's assume we need to rotate to 'ninety_deg'
        #    (In a real scenario, angle names and values would be mapped from the environment)
        angle_from = 'zero_deg'
        angle_to = 'ninety_deg'
        try:
            print(f"[Exploration] Rotating gripper from {angle_from} to {angle_to}.")
            obs, reward, done = rotate(
                env,
                task,
                gripper=drawer_name,  # This is a placeholder; actual gripper name may differ
                from_angle=angle_from,
                to_angle=angle_to
            )
            if done:
                print("[Exploration] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during rotate: {e}")
            # Continue; rotation may not be strictly required for exploration

        # 3. Attempt to interact with the drawer (e.g., pull) to see if side position is sufficient
        try:
            print(f"[Exploration] Attempting to pull drawer '{drawer_name}'.")
            obs, reward, done = pull(
                env,
                task,
                gripper=drawer_name,  # Placeholder; actual gripper name may differ
                drawer=drawer_name
            )
            if done:
                print("[Exploration] Task ended during pull!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pull: {e}")
            # This may fail if preconditions (e.g., holding-drawer) are not met

        # 4. If the above fails, the missing predicate (is-side-pos pos0 drawer0) is likely required
        print("[Exploration] If previous actions failed, (is-side-pos pos0 drawer0) is likely missing or not satisfied.")

        # === End of Exploration Phase ===

        # === (Optional) Continue with Oracle Plan if exploration is successful ===
        # Here you would continue with the main plan using the available skills:
        # e.g., move to anchor position, pick-drawer, pull, etc.
        # For this task, we focus on the exploration phase as per feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
