# run_skeleton_task.py (Completed with Exploration Phase and Skill Calls)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer': ..., 'gripper': ..., 'pos1': ..., 'pos2': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Feedback: (is-side-pos pos2 ?d)
        # The missing predicate is (is-side-pos pos2 ?d)
        # We need to explore and check if pos2 is a side position for the drawer (?d)
        # This is typically done by attempting to use the move-to-side action or by checking the environment

        # For demonstration, let's assume we have a drawer and two positions: pos1 (anchor), pos2 (side)
        # We'll attempt to move the gripper to pos2 and see if the action is possible (i.e., if is-side-pos holds)
        # If not, we log the missing predicate

        # --- Example object/position names (replace with actual names from your environment) ---
        drawer_name = 'drawer'
        gripper_name = 'gripper'
        pos1_name = 'pos1'  # anchor position
        pos2_name = 'pos2'  # side position

        # Get positions for gripper, pos1, pos2
        try:
            gripper_pos = positions[gripper_name]
            pos1 = positions[pos1_name]
            pos2 = positions[pos2_name]
            drawer = positions[drawer_name]
        except KeyError as e:
            print(f"[Exploration] Missing object or position in positions dict: {e}")
            shutdown_environment(env)
            return

        # --- Exploration: Try to move to pos2 as a side position ---
        print("[Exploration] Attempting to move gripper to pos2 (side position) to test (is-side-pos pos2 ?d) ...")
        try:
            # Move gripper to pos2 (side position)
            obs, reward, done = move(env, task, target_pos=pos2)
            if done:
                print("[Exploration] Task ended unexpectedly during exploration move.")
                return
            # If move is successful, we can assume (is-side-pos pos2 ?d) holds
            print("[Exploration] Successfully moved to pos2. (is-side-pos pos2 ?d) likely holds.")
        except Exception as e:
            print(f"[Exploration] Failed to move to pos2: {e}")
            print("[Exploration] (is-side-pos pos2 ?d) predicate may be missing or not satisfied.")
            # Log or handle missing predicate as needed
            shutdown_environment(env)
            return

        # === Main Task Plan Execution ===
        # Example: Open the drawer using the available skills

        # 1. Move gripper to anchor position (pos1)
        print("[Task] Moving gripper to anchor position (pos1)...")
        try:
            obs, reward, done = move(env, task, target_pos=pos1)
            if done:
                print("[Task] Task ended after moving to anchor position!")
                return
        except Exception as e:
            print(f"[Task] Error during move to anchor: {e}")
            shutdown_environment(env)
            return

        # 2. Rotate gripper to required angle (e.g., ninety_deg)
        # For demonstration, assume we have angle values in positions or a separate dict
        angle_from = 'zero_deg'
        angle_to = 'ninety_deg'
        print("[Task] Rotating gripper from zero_deg to ninety_deg...")
        try:
            obs, reward, done = rotate(env, task, gripper_name, angle_from, angle_to)
            if done:
                print("[Task] Task ended after rotation!")
                return
        except Exception as e:
            print(f"[Task] Error during rotate: {e}")
            shutdown_environment(env)
            return

        # 3. Move gripper to side position (pos2)
        print("[Task] Moving gripper to side position (pos2)...")
        try:
            obs, reward, done = move(env, task, target_pos=pos2)
            if done:
                print("[Task] Task ended after moving to side position!")
                return
        except Exception as e:
            print(f"[Task] Error during move to side: {e}")
            shutdown_environment(env)
            return

        # 4. Move to anchor position (simulate move-to-anchor)
        print("[Task] Moving gripper back to anchor position (pos1)...")
        try:
            obs, reward, done = move(env, task, target_pos=pos1)
            if done:
                print("[Task] Task ended after moving to anchor position!")
                return
        except Exception as e:
            print(f"[Task] Error during move to anchor: {e}")
            shutdown_environment(env)
            return

        # 5. Pick the drawer handle (simulate pick-drawer)
        print("[Task] Picking the drawer handle at anchor position...")
        try:
            obs, reward, done = pick(env, task, drawer_name, pos1_name)
            if done:
                print("[Task] Task ended after picking drawer handle!")
                return
        except Exception as e:
            print(f"[Task] Error during pick: {e}")
            shutdown_environment(env)
            return

        # 6. Pull the drawer open
        print("[Task] Pulling the drawer open...")
        try:
            obs, reward, done = pull(env, task, drawer_name)
            if done:
                print("[Task] Task ended after pulling drawer!")
                return
        except Exception as e:
            print(f"[Task] Error during pull: {e}")
            shutdown_environment(env)
            return

        # 7. Place an object if required (example, not always needed)
        # print("[Task] Placing object...")
        # try:
        #     obs, reward, done = place(env, task, object_name, target_location)
        #     if done:
        #         print("[Task] Task ended after placing object!")
        #         return
        # except Exception as e:
        #     print(f"[Task] Error during place: {e}")
        #     shutdown_environment(env)
        #     return

        print("[Task] Task plan executed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()