# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Example expected keys: 'drawer1', 'drawer1_handle', 'drawer1_side', 'drawer1_anchor', etc.
        # These keys must match your environment's naming conventions.

        # --- Exploration Phase: Identify Missing Predicate ---
        # Feedback indicates (is-open drawer1) is missing or needs to be explored.
        # We will attempt to open the drawer and check its state.

        # 1. Move gripper to the side position of drawer1 (for alignment)
        try:
            drawer1_side_pos = positions['drawer1_side']
            print("[Exploration] Moving to drawer1 side position:", drawer1_side_pos)
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(drawer1_side_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended during move to side position!")
                return
        except Exception as e:
            print("[Error] Could not move to drawer1 side position:", e)
            return

        # 2. Rotate gripper to 90 degrees (if required for side approach)
        try:
            print("[Exploration] Rotating gripper to 90 degrees for side approach.")
            obs, reward, done = rotate(
                env,
                task,
                angle='ninety_deg'  # Assumes 'ninety_deg' is a valid angle identifier in your environment
            )
            if done:
                print("[Exploration] Task ended during gripper rotation!")
                return
        except Exception as e:
            print("[Error] Could not rotate gripper:", e)
            return

        # 3. Move gripper to anchor position (for grasping drawer handle)
        try:
            drawer1_anchor_pos = positions['drawer1_anchor']
            print("[Exploration] Moving to drawer1 anchor position:", drawer1_anchor_pos)
            obs, reward, done = move(
                env,
                task,
                target_pos=np.array(drawer1_anchor_pos),
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended during move to anchor position!")
                return
        except Exception as e:
            print("[Error] Could not move to drawer1 anchor position:", e)
            return

        # 4. Pick (grasp) the drawer handle at anchor position
        try:
            print("[Exploration] Grasping drawer1 handle at anchor position.")
            obs, reward, done = pick(
                env,
                task,
                target_pos=np.array(drawer1_anchor_pos),
                approach_distance=0.10,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended during pick (grasp)!")
                return
        except Exception as e:
            print("[Error] Could not pick (grasp) drawer1 handle:", e)
            return

        # 5. Pull the drawer to attempt to open it (this should set (is-open drawer1))
        try:
            print("[Exploration] Pulling drawer1 to open.")
            obs, reward, done = pull(
                env,
                task,
                drawer_name='drawer1'
            )
            if done:
                print("[Exploration] Task ended during pull (open)!")
                return
        except Exception as e:
            print("[Error] Could not pull (open) drawer1:", e)
            return

        # 6. Check if the drawer is open (predicate (is-open drawer1) should now be true)
        #    This is typically done by querying the environment or checking the observation.
        try:
            obs = task.get_observation()
            # The way to check if the drawer is open depends on your observation structure.
            # For example, you might have: obs.drawer1_is_open or similar.
            is_open = getattr(obs, 'drawer1_is_open', None)
            if is_open is not None:
                print(f"[Exploration] drawer1 is_open: {is_open}")
            else:
                print("[Exploration] Could not determine if drawer1 is open from observation.")
        except Exception as e:
            print("[Error] Could not check drawer1 open state:", e)

        print("[Exploration] Exploration phase complete. (is-open drawer1) predicate should now be known.")

        # --- End of Exploration Phase ---

        # You can now proceed with the rest of your oracle plan, using the available skills,
        # and knowing that (is-open drawer1) can be achieved by the above sequence.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()