# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'gripper': ..., 'approach_bin_pos': ..., ...}
        positions = get_object_positions()

        # === Exploration Phase: Check for Missing Predicate ===
        # Feedback: (gripper-at gripper approach_bin_pos) is missing.
        # We will check if the gripper is at 'approach_bin_pos' and, if not, move it there.

        gripper_name = 'gripper'
        approach_bin_pos_name = 'approach_bin_pos'

        # Defensive: Check if required keys exist in positions
        if gripper_name not in positions or approach_bin_pos_name not in positions:
            print(f"[Exploration] ERROR: Required keys '{gripper_name}' or '{approach_bin_pos_name}' not found in object positions.")
            return

        approach_bin_pos = positions[approach_bin_pos_name]

        # Get current gripper position from observation
        try:
            obs = task.get_observation()
            current_gripper_pos = obs.gripper_pose[:3]
        except Exception as e:
            print(f"[Exploration] ERROR: Could not get gripper position from observation: {e}")
            return

        # Check if gripper is already at approach_bin_pos (within a small threshold)
        dist = np.linalg.norm(np.array(current_gripper_pos) - np.array(approach_bin_pos))
        threshold = 0.02  # 2cm tolerance

        if dist > threshold:
            print(f"[Exploration] Gripper is NOT at approach_bin_pos (distance: {dist:.3f}). Moving gripper there.")
            try:
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=np.array(approach_bin_pos),
                    max_steps=100,
                    threshold=0.01,
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended during move to approach_bin_pos!")
                    return
            except Exception as e:
                print(f"[Exploration] ERROR during move: {e}")
                return
        else:
            print(f"[Exploration] Gripper is already at approach_bin_pos (distance: {dist:.3f}).")

        # After moving, confirm the predicate is now true (for exploration/diagnostics)
        obs = task.get_observation()
        new_gripper_pos = obs.gripper_pose[:3]
        new_dist = np.linalg.norm(np.array(new_gripper_pos) - np.array(approach_bin_pos))
        if new_dist < threshold:
            print("[Exploration] SUCCESS: (gripper-at gripper approach_bin_pos) is now true.")
        else:
            print("[Exploration] WARNING: Gripper did not reach approach_bin_pos as expected.")

        # === (Optional) Continue with further plan steps as needed ===
        # For this exploration, we only demonstrate the missing predicate handling.
        # Additional plan steps (pick, place, rotate, pull, etc.) would be added here as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()