# run_skeleton_task.py (Completed Executable)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # Example usage: positions = {'drawer0': ..., 'object_1': ..., ...}
        positions = get_object_positions()

        # --- Exploration Phase: Identify Missing Predicate (is-open drawer0) ---
        # Feedback indicates that (is-open drawer0) is missing or needs to be explored.
        # We will attempt to open the drawer using available skills and check the result.

        # Assumptions:
        # - The drawer is named 'drawer0'
        # - The gripper is named 'gripper0'
        # - The drawer has anchor and side positions defined in positions dict
        # - The angles are named 'zero_deg' and 'ninety_deg'
        # - The robot starts with handempty and at a default position

        # Extract required positions and parameters
        try:
            drawer_name = 'drawer0'
            gripper_name = 'gripper0'
            anchor_pos_name = f'{drawer_name}_anchor'
            side_pos_name = f'{drawer_name}_side'
            zero_deg_name = 'zero_deg'
            ninety_deg_name = 'ninety_deg'

            # Get positions for gripper to move
            anchor_pos = positions[anchor_pos_name]
            side_pos = positions[side_pos_name]
            gripper_home = positions.get('gripper_home', None)
            if gripper_home is None:
                gripper_home = positions[side_pos_name]  # fallback

            # Step 1: Move gripper to side position (pre-approach)
            print(f"[Exploration] Moving gripper to side position: {side_pos_name} at {side_pos}")
            obs, reward, done = move(env, task, target_pos=side_pos)
            if done:
                print("[Exploration] Task ended unexpectedly during move to side position.")
                return

            # Step 2: Rotate gripper to 90 degrees (if not already)
            print(f"[Exploration] Rotating gripper to ninety_deg")
            try:
                obs, reward, done = rotate(env, task, gripper_name, zero_deg_name, ninety_deg_name)
                if done:
                    print("[Exploration] Task ended unexpectedly during rotate.")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during rotate: {e}")

            # Step 3: Move gripper to anchor position (for grasping drawer handle)
            print(f"[Exploration] Moving gripper to anchor position: {anchor_pos_name} at {anchor_pos}")
            obs, reward, done = move(env, task, target_pos=anchor_pos)
            if done:
                print("[Exploration] Task ended unexpectedly during move to anchor position.")
                return

            # Step 4: Pick the drawer handle (pick-drawer is not in available skills, so skip)
            # Instead, try to pull directly if possible, or simulate grasp if required

            # Step 5: Pull the drawer open
            print(f"[Exploration] Attempting to pull the drawer open: {drawer_name}")
            try:
                obs, reward, done = pull(env, task, gripper_name, drawer_name)
                if done:
                    print("[Exploration] Task ended unexpectedly during pull.")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during pull: {e}")

            # Step 6: Check if the drawer is open (predicate is-open drawer0)
            # This would be checked in the environment or by observation
            # For demonstration, print feedback
            print("[Exploration] Exploration complete. Check if (is-open drawer0) is now true.")

        except KeyError as e:
            print(f"[Exploration] Missing position key: {e}")
        except Exception as e:
            print(f"[Exploration] Unexpected error during exploration: {e}")

        # --- End of Exploration Phase ---

        # You can add further plan steps here as needed, using only the available skills:
        # pick, place, move, rotate, pull

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()