# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # Based on feedback and exploration knowledge, we need to check for missing predicates
        # such as 'weight-known', 'durability-known', 'lock-known', etc.
        # We'll use available skills to explore and update our knowledge.

        # For demonstration, let's assume we have a list of objects and locations
        # (In practice, you would extract these from the observation or positions)
        # We'll try to pick each object to see if we can learn its weight/durability,
        # and try to pull drawers to learn their lock state.

        # Example object and location lists (replace with actual from positions/obs)
        object_names = [name for name in positions.keys() if 'object' in name or 'drawer' in name]
        location_names = set()
        for obj in object_names:
            pos_info = positions[obj]
            if isinstance(pos_info, dict) and 'location' in pos_info:
                location_names.add(pos_info['location'])
        location_names = list(location_names)
        if not location_names:
            # Fallback: try to infer locations from positions
            location_names = ['location1', 'location2']

        # For the sake of exploration, try to pick and pull each object if possible
        for obj in object_names:
            try:
                # Try to get the location of the object
                obj_pos = positions[obj]
                if isinstance(obj_pos, dict) and 'location' in obj_pos:
                    obj_loc = obj_pos['location']
                else:
                    # Fallback: use a default or first location
                    obj_loc = location_names[0]
                
                print(f"[Exploration] Attempting to pick {obj} at {obj_loc}")
                # Try to pick the object (if possible)
                obs, reward, done = pick(
                    env,
                    task,
                    obj,
                    obj_loc
                )
                if done:
                    print(f"[Exploration] Task ended after picking {obj}!")
                    return

                # After picking, try to pull if it's a drawer
                if 'drawer' in obj:
                    print(f"[Exploration] Attempting to pull {obj} at {obj_loc}")
                    obs, reward, done = pull(
                        env,
                        task,
                        obj,
                        obj_loc
                    )
                    if done:
                        print(f"[Exploration] Task ended after pulling {obj}!")
                        return

                # Place the object back (if possible)
                print(f"[Exploration] Attempting to place {obj} at {obj_loc}")
                obs, reward, done = place(
                    env,
                    task,
                    obj,
                    obj_loc
                )
                if done:
                    print(f"[Exploration] Task ended after placing {obj}!")
                    return

            except Exception as e:
                print(f"[Exploration] Exception during exploration of {obj}: {e}")
                continue

        # === End of Exploration Phase ===

        # At this point, the exploration should have triggered any missing predicates
        # (e.g., weight-known, durability-known, lock-known) via the effects of pick/pull.

        print("[Exploration] Exploration phase complete. If any missing predicates existed, they should now be discovered.")

        # === Main Task Plan (if any) ===
        # Here you would execute the actual oracle plan using the available skills.
        # For this code, we focus on the exploration phase as per the feedback.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
