# run_skeleton_task.py (Completed with Exploration Phase and Error Handling)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The goal is to discover which predicate is missing (e.g., lock-known, temperature-known, etc.)
        # We will attempt to use each available skill and observe the effect/state.

        available_skills = ['pick', 'place', 'move', 'rotate', 'pull']
        exploration_results = {}

        # For demonstration, we will try to use each skill on available objects/positions
        # and check for errors or new information in the observation.

        # Get a list of objects and locations from positions
        object_names = [name for name in positions if not name.startswith('location_')]
        location_names = [name for name in positions if name.startswith('location_')]

        # Use the first object and location as test targets if available
        test_object = object_names[0] if object_names else None
        test_location = location_names[0] if location_names else None

        # 1. Try 'move' skill (if implemented in skill_code)
        if 'move' in available_skills and test_location is not None:
            try:
                print(f"[Exploration] Trying move to {test_location}")
                obs, reward, done = move(env, task, target_pos=positions[test_location])
                exploration_results['move'] = 'success'
            except Exception as e:
                print(f"[Exploration] move failed: {e}")
                exploration_results['move'] = f'error: {e}'

        # 2. Try 'pick' skill
        if 'pick' in available_skills and test_object is not None and test_location is not None:
            try:
                print(f"[Exploration] Trying pick on {test_object} at {test_location}")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=positions[test_object],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                exploration_results['pick'] = 'success'
            except Exception as e:
                print(f"[Exploration] pick failed: {e}")
                exploration_results['pick'] = f'error: {e}'

        # 3. Try 'rotate' skill (if gripper or object supports rotation)
        if 'rotate' in available_skills:
            try:
                print(f"[Exploration] Trying rotate")
                # Use a default quaternion for demonstration (identity quaternion)
                target_quat = np.array([0, 0, 0, 1])
                obs, reward, done = rotate(
                    env,
                    task,
                    target_quat=target_quat,
                    max_steps=50,
                    threshold=0.05,
                    timeout=5.0
                )
                exploration_results['rotate'] = 'success'
            except Exception as e:
                print(f"[Exploration] rotate failed: {e}")
                exploration_results['rotate'] = f'error: {e}'

        # 4. Try 'pull' skill (if object supports pulling, e.g., drawer)
        if 'pull' in available_skills and test_object is not None:
            try:
                print(f"[Exploration] Trying pull on {test_object}")
                obs, reward, done = pull(env, task, target_name=test_object)
                exploration_results['pull'] = 'success'
            except Exception as e:
                print(f"[Exploration] pull failed: {e}")
                exploration_results['pull'] = f'error: {e}'

        # 5. Try 'place' skill (if holding an object)
        if 'place' in available_skills and test_object is not None and test_location is not None:
            try:
                print(f"[Exploration] Trying place {test_object} at {test_location}")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=positions[test_location],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                exploration_results['place'] = 'success'
            except Exception as e:
                print(f"[Exploration] place failed: {e}")
                exploration_results['place'] = f'error: {e}'

        print("[Exploration] Results:", exploration_results)

        # === End of Exploration Phase ===

        # === Main Task Plan (Oracle Plan Execution) ===
        # Here, you would execute the actual plan to achieve the goal, using only the available skills.
        # For demonstration, we show a generic sequence (replace with your actual plan as needed):

        # Example: Pick up the first object and place it at the first location
        if test_object is not None and test_location is not None:
            try:
                print(f"[Task] Picking {test_object} at {test_location}")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=positions[test_object],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after pick!")
                    return

                print(f"[Task] Placing {test_object} at {test_location}")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=positions[test_location],
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Task] Task ended after place!")
                    return

            except Exception as e:
                print(f"[Task] Error during pick/place: {e}")

        # You can expand this section to include more complex plans as needed.

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()