# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def is_object_in_env(obj_name, positions):
    """Check if the object is present in the environment."""
    return obj_name in positions

def is_gripper_safe(env, task, target_pos, min_distance=0.05):
    """Check if the gripper is at a safe distance from the target position."""
    obs = task.get_observation()
    gripper_pos = obs.gripper_pose[:3]
    dist = np.linalg.norm(np.array(gripper_pos) - np.array(target_pos))
    return dist >= min_distance

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        object_list = list(positions.keys())

        # === Exploration Phase: Identify Missing Predicate ===
        # Try to pick each object and check for errors or missing knowledge
        # This phase is to discover if any predicate (e.g., identified, temperature-known, etc.) is missing
        print("[Exploration] Starting exploration to identify missing predicates...")
        for obj_name in object_list:
            try:
                # Check object presence
                if not is_object_in_env(obj_name, positions):
                    print(f"[Exploration] Object '{obj_name}' not found in environment. Skipping.")
                    continue

                obj_pos = positions[obj_name]
                # Safety check: is gripper at a safe distance?
                if not is_gripper_safe(env, task, obj_pos):
                    print(f"[Exploration] Gripper too close to '{obj_name}'. Skipping for safety.")
                    continue

                print(f"[Exploration] Attempting to pick '{obj_name}' at position {obj_pos}...")
                # Try to pick the object using the provided skill
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after picking '{obj_name}'.")
                    break

                # If pick succeeded, try to place it back (if possible)
                print(f"[Exploration] Attempting to place '{obj_name}' back at position {obj_pos}...")
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Exploration] Task ended after placing '{obj_name}'.")
                    break

            except Exception as e:
                print(f"[Exploration] Exception occurred while handling '{obj_name}': {e}")
                continue

        print("[Exploration] Exploration phase complete. If errors occurred, check for missing predicates in the domain.")

        # === Main Task Plan (Example) ===
        # After exploration, proceed with the main plan using only available skills
        # For demonstration, pick and place the first object safely

        for obj_name in object_list:
            try:
                obj_pos = positions[obj_name]
                if not is_object_in_env(obj_name, positions):
                    print(f"[Task] Object '{obj_name}' not found. Skipping.")
                    continue
                if not is_gripper_safe(env, task, obj_pos):
                    print(f"[Task] Gripper too close to '{obj_name}'. Skipping for safety.")
                    continue

                print(f"[Task] Picking '{obj_name}' at {obj_pos}...")
                obs, reward, done = pick(
                    env,
                    task,
                    target_pos=obj_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print(f"[Task] Task ended after picking '{obj_name}'.")
                    return

                # Example: Place at a bin if available
                bin_name = None
                for candidate in object_list:
                    if "bin" in candidate:
                        bin_name = candidate
                        break
                if bin_name is not None:
                    bin_pos = positions[bin_name]
                    print(f"[Task] Placing '{obj_name}' into '{bin_name}' at {bin_pos}...")
                    obs, reward, done = place(
                        env,
                        task,
                        target_pos=bin_pos,
                        approach_distance=0.15,
                        max_steps=100,
                        threshold=0.01,
                        approach_axis='z',
                        timeout=10.0
                    )
                    if done:
                        print(f"[Task] Task ended after placing '{obj_name}' into '{bin_name}'.")
                        return
                else:
                    print(f"[Task] No bin found to place '{obj_name}'. Skipping place step.")

            except Exception as e:
                print(f"[Task] Exception occurred while handling '{obj_name}': {e}")
                continue

        print("===== Task completed successfully. =====")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()