# run_skeleton_task.py (Completed with Exploration Phase and Safety/Predicate Checks)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

import time

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation, with exploration for missing predicates.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Example: positions = {'tomato': (x, y, z), 'bin': (x, y, z), ...}

        # === Object List Verification ===
        # Only use objects that are present in the environment
        available_objects = list(positions.keys())
        print("[Info] Available objects in environment:", available_objects)

        # === Safety and Predicate Exploration Phase ===
        # The goal is to discover if any required predicate (e.g., lock-known, identified, etc.) is missing
        # and to ensure safe manipulation (e.g., not pulling a locked drawer).
        # We will use the available skills to explore and check for missing predicates.

        # For demonstration, let's assume we have a drawer and a tomato to manipulate.
        # We'll try to open the drawer, pick the tomato, and place it in the bin.
        # We'll check for lock status, object identification, and safe rotation.

        # --- Step 1: Identify and Check Drawer Lock Status ---
        # Find drawer and gripper if present
        drawer_name = None
        gripper_name = None
        tomato_name = None
        bin_name = None
        for obj in available_objects:
            if 'drawer' in obj:
                drawer_name = obj
            elif 'gripper' in obj:
                gripper_name = obj
            elif 'tomato' in obj:
                tomato_name = obj
            elif 'bin' in obj or 'disposal' in obj:
                bin_name = obj

        # Safety: Ensure all required objects are present
        if not all([drawer_name, gripper_name, tomato_name, bin_name]):
            print("[Error] Required objects (drawer, gripper, tomato, bin) not found in environment!")
            return

        # --- Step 2: Exploration - Check if drawer is locked ---
        # Try to pull the drawer; if it fails, assume it's locked and needs to be unlocked or skipped
        print(f"[Exploration] Checking if {drawer_name} is locked by attempting to pull.")
        try:
            obs, reward, done = pull(env, task, gripper_name, drawer_name)
            if done:
                print("[Exploration] Task ended during drawer pull!")
                return
            print(f"[Exploration] Successfully pulled {drawer_name} (drawer is unlocked).")
        except Exception as e:
            print(f"[Exploration] Exception during pull: {e}")
            print(f"[Exploration] {drawer_name} may be locked or not grasped. Skipping pull or attempting unlock if possible.")

        # --- Step 3: Exploration - Rotate Gripper Safely ---
        # Use rotate skill to set gripper to a safe orientation (e.g., 90 degrees)
        # For demonstration, use a target quaternion for 90 degrees rotation about z-axis
        print("[Exploration] Rotating gripper to safe orientation (90 degrees about z).")
        try:
            # Example quaternion for 90 degrees about z: [0, 0, sqrt(0.5), sqrt(0.5)]
            target_quat = np.array([0, 0, np.sqrt(0.5), np.sqrt(0.5)])
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Exploration] Task ended during rotation!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during rotate: {e}")

        # --- Step 4: Exploration - Identify Tomato and Check Properties ---
        # Move to tomato location to identify it (simulate 'identified' predicate)
        tomato_pos = positions[tomato_name]
        print(f"[Exploration] Moving to {tomato_name} at {tomato_pos} to identify.")
        try:
            obs, reward, done = move(env, task, target_pos=tomato_pos)
            if done:
                print("[Exploration] Task ended during move to tomato!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during move: {e}")

        # --- Step 5: Exploration - Pick Tomato and Check Weight/Durability ---
        print(f"[Exploration] Picking up {tomato_name}.")
        try:
            obs, reward, done = pick(env, task, target_pos=tomato_pos)
            if done:
                print("[Exploration] Task ended during pick!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pick: {e}")

        # --- Step 6: Exploration - Move to Bin and Place Tomato ---
        bin_pos = positions[bin_name]
        print(f"[Exploration] Moving to bin at {bin_pos}.")
        try:
            obs, reward, done = move(env, task, target_pos=bin_pos)
            if done:
                print("[Exploration] Task ended during move to bin!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during move to bin: {e}")

        print(f"[Exploration] Placing {tomato_name} in bin.")
        try:
            obs, reward, done = place(env, task, target_pos=bin_pos)
            if done:
                print("[Exploration] Task ended after placing tomato!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during place: {e}")

        # --- Step 7: Final Safety and Completion Check ---
        print("[Task] Checking for any remaining safety or completion conditions.")
        # Here, you could check for any remaining objects, open drawers, or other safety predicates
        # For now, we assume the task is complete if no exceptions occurred

        print("[Task] Task completed successfully.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
