# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def euler_from_quat(quaternion):
    """
    Convert a quaternion to Euler angles (roll, pitch, yaw).
    Parameters:
    quaternion (numpy array): A quaternion (4-element array, [x, y, z, w]).
    Returns:
    Euler angles (numpy array): The Euler angles (3-element array).
    """
    x, y, z, w = quaternion
    # roll (x-axis rotation)
    t0 = +2.0 * (w * x + y * z)
    t1 = +1.0 - 2.0 * (x * x + y * y)
    roll_x = np.arctan2(t0, t1)
    # pitch (y-axis rotation)
    t2 = +2.0 * (w * y - z * x)
    t2 = +1.0 if t2 > +1.0 else t2
    t2 = -1.0 if t2 < -1.0 else t2
    pitch_y = np.arcsin(t2)
    # yaw (z-axis rotation)
    t3 = +2.0 * (w * z + x * y)
    t4 = +1.0 - 2.0 * (y * y + z * z)
    yaw_z = np.arctan2(t3, t4)
    return np.array([roll_x, pitch_y, yaw_z])

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # For demonstration, print all available objects and their positions
        print("[Info] Available objects and positions:", positions)

        # === Robust Object Existence Check ===
        # Let's say our plan needs to manipulate 'object_1' and 'object_2'
        required_objects = ['object_1', 'object_2']
        for obj in required_objects:
            if obj not in positions:
                print(f"[Warning] Required object '{obj}' not found in environment! Skipping.")
        
        # === Example Plan Execution ===
        # This is a placeholder plan. Replace with your actual oracle plan steps.
        # We'll demonstrate pick, place, move, rotate, pull, and exploration for missing predicates.

        # 1. Move to object_1's location (if available)
        if 'object_1' in positions:
            object_1_pos = positions['object_1']
            try:
                print(f"[Task] Moving to object_1 at {object_1_pos}")
                obs, reward, done = move(env, task, target_pos=object_1_pos)
                if done:
                    print("[Task] Task ended after move!")
                    return
            except Exception as e:
                print(f"[Error] Exception during move to object_1: {e}")

            # 2. Pick object_1
            try:
                print(f"[Task] Picking object_1 at {object_1_pos}")
                obs, reward, done = pick(env, task, target_pos=object_1_pos)
                if done:
                    print("[Task] Task ended after pick!")
                    return
            except Exception as e:
                print(f"[Error] Exception during pick of object_1: {e}")

            # 3. Place object_1 at a new location (if available)
            if 'target_location' in positions:
                target_location = positions['target_location']
                try:
                    print(f"[Task] Placing object_1 at {target_location}")
                    obs, reward, done = place(env, task, target_pos=target_location)
                    if done:
                        print("[Task] Task ended after place!")
                        return
                except Exception as e:
                    print(f"[Error] Exception during place of object_1: {e}")

        # 4. Rotate gripper to a specific orientation (demonstration)
        # Let's use a target quaternion (e.g., [0, 0, 0, 1] for no rotation)
        target_quat = np.array([0, 0, 0, 1])
        try:
            print(f"[Task] Rotating gripper to {target_quat}")
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Error] Exception during rotate: {e}")

        # 5. Pull action (if available and meaningful in your environment)
        # For demonstration, try pulling 'object_2' if it exists
        if 'object_2' in positions:
            object_2_pos = positions['object_2']
            try:
                print(f"[Task] Moving to object_2 at {object_2_pos}")
                obs, reward, done = move(env, task, target_pos=object_2_pos)
                if done:
                    print("[Task] Task ended after move to object_2!")
                    return
            except Exception as e:
                print(f"[Error] Exception during move to object_2: {e}")

            try:
                print(f"[Task] Pulling object_2 at {object_2_pos}")
                obs, reward, done = pull(env, task, target_pos=object_2_pos)
                if done:
                    print("[Task] Task ended after pull!")
                    return
            except Exception as e:
                print(f"[Error] Exception during pull of object_2: {e}")

        # === Exploration Phase for Missing Predicate ===
        # Based on exploration knowledge, try to identify missing predicates by performing actions and observing state changes.
        # For example, after pick, check if 'weight-known' or 'durability-known' is set.
        # Here, we simply print out the observation after each action for manual inspection.

        print("[Exploration] Checking for missing predicates in observation...")
        obs = task.get_observation()
        # Print all attributes of obs for inspection
        for attr in dir(obs):
            if not attr.startswith('_'):
                value = getattr(obs, attr)
                print(f"  [Obs] {attr}: {value}")

        # If you want to check for a specific predicate, e.g., 'weight-known', do:
        # if hasattr(obs, 'weight_known'):
        #     print("[Exploration] weight-known predicate detected in observation.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()
