# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only predefined skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Exploration Phase: Identify Missing Predicate ===
        # The goal is to determine which predicate is missing (e.g., lock-known, identified, etc.)
        # We'll use available skills to interact with the environment and infer missing information.

        # For demonstration, let's assume we have a drawer and an object to interact with.
        # We'll try to pick an object and pull a drawer, observing the effects.

        # Example object and drawer names (replace with actual names from your environment)
        # These should be keys in the positions dictionary.
        drawer_name = None
        object_name = None
        drawer_handle_pos = None
        object_pos = None

        # Try to find a drawer and an object in the positions dictionary
        for name in positions:
            if 'drawer' in name and drawer_name is None:
                drawer_name = name
                drawer_handle_pos = positions[name]
            elif 'object' in name or 'ball' in name or 'tomato' in name:
                object_name = name
                object_pos = positions[name]
            if drawer_name and object_name:
                break

        if not drawer_name or not object_name:
            print("[Exploration] Could not find both a drawer and an object in the environment.")
            return

        print(f"[Exploration] Using drawer: {drawer_name} at {drawer_handle_pos}")
        print(f"[Exploration] Using object: {object_name} at {object_pos}")

        # 1. Try to pick the object (to check for weight-known/durability-known predicates)
        try:
            print(f"[Exploration] Attempting to pick {object_name} at {object_pos}")
            obs, reward, done = pick(
                env,
                task,
                target_pos=object_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended after pick!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pick: {e}")

        # 2. Try to place the object back (if possible)
        try:
            print(f"[Exploration] Attempting to place {object_name} at {object_pos}")
            obs, reward, done = place(
                env,
                task,
                target_pos=object_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended after place!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during place: {e}")

        # 3. Try to rotate the gripper (to check for rotated predicate)
        try:
            print(f"[Exploration] Attempting to rotate gripper to 90 degrees (example quaternion)")
            # Example: rotate to 90 degrees about z-axis
            from scipy.spatial.transform import Rotation as R
            quat_90deg = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
            # RLBench may expect wxyz, so swap if needed
            quat_90deg = np.roll(quat_90deg, 1)
            obs, reward, done = rotate(
                env,
                task,
                target_quat=quat_90deg,
                max_steps=100,
                threshold=0.05,
                timeout=10.0
            )
            if done:
                print("[Exploration] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during rotate: {e}")

        # 4. Try to pull the drawer (to check for is-locked/is-open predicates)
        try:
            print(f"[Exploration] Attempting to pull {drawer_name}")
            obs, reward, done = pull(
                env,
                task,
                drawer_name
            )
            if done:
                print("[Exploration] Task ended after pull!")
                return
        except Exception as e:
            print(f"[Exploration] Exception during pull: {e}")

        # 5. Try to move (if move skill is available and implemented)
        if 'move' in globals():
            try:
                print(f"[Exploration] Attempting to move (if implemented)")
                # Example: move to the object's position (if move skill expects positions)
                obs, reward, done = move(
                    env,
                    task,
                    target_pos=object_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
                if done:
                    print("[Exploration] Task ended after move!")
                    return
            except Exception as e:
                print(f"[Exploration] Exception during move: {e}")

        print("[Exploration] Exploration phase complete. Check logs and environment state to infer missing predicates.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()