import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Object name mapping (from object list and environment)
        # Drawer positions
        bottom_anchor_pos = positions.get('bottom_anchor_pos')
        bottom_side_pos = positions.get('bottom_side_pos')
        bottom_joint_pos = positions.get('bottom_joint_pos')
        middle_anchor_pos = positions.get('middle_anchor_pos')
        middle_side_pos = positions.get('middle_side_pos')
        middle_joint_pos = positions.get('middle_joint_pos')
        top_anchor_pos = positions.get('top_anchor_pos')
        top_side_pos = positions.get('top_side_pos')
        top_joint_pos = positions.get('top_joint_pos')
        # Items (tomatoes)
        item1_pos = positions.get('item1')
        item2_pos = positions.get('item2')
        item3_pos = positions.get('item3')
        # Plate
        plate_pos = positions.get('plate')

        # === Plan Execution ===
        # Goal: Select a drawer and open it fully, then pick up all the tomatoes and leave them on the plate.

        # For this plan, we will use the "bottom" drawer as the selected drawer.
        # The plan is as follows (based on the provided specification steps):

        # Step 1: Move gripper to the side position of the bottom drawer
        print("[Step 1] Move to bottom_side_pos")
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Task ended at step 1!")
            return

        # Step 2: Rotate gripper to 90 degrees (for drawer grasping)
        print("[Step 2] Rotate gripper to 90 degrees")
        # Assume that the target orientation for 90 degrees is provided in positions or can be constructed
        # If not, we can try to get the current orientation and rotate around z by 90 deg
        # For simplicity, let's try to get the orientation from positions if available, else compute
        # We'll use scipy.spatial.transform.R to rotate
        from scipy.spatial.transform import Rotation as R
        obs_now = task.get_observation()
        current_quat = obs_now.gripper_pose[3:7]
        # Rotate 90 deg around z axis
        r = R.from_quat(current_quat)
        r_90 = r * R.from_euler('z', 90, degrees=True)
        target_quat_90 = r_90.as_quat()
        obs, reward, done = rotate(env, task, target_quat_90)
        if done:
            print("[Task] Task ended at step 2!")
            return

        # Step 3: Move to anchor position of the bottom drawer (for grasping handle)
        print("[Step 3] Move to bottom_anchor_pos")
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Task ended at step 3!")
            return

        # Step 4: Pick (grasp) the drawer handle
        print("[Step 4] Pick (grasp) the bottom drawer handle")
        # For picking the drawer handle, use the anchor position
        obs, reward, done = pick(env, task, bottom_anchor_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended at step 4!")
            return

        # Step 5: Pull the drawer open (along x axis, positive direction)
        print("[Step 5] Pull the bottom drawer open")
        # Estimate pull distance (e.g., 0.20m for full open, adjust as needed)
        pull_distance = 0.20
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Task] Task ended at step 5!")
            return

        # Step 6: Move gripper away from drawer (to side position)
        print("[Step 6] Move to bottom_side_pos (clear drawer)")
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Task ended at step 6!")
            return

        # Step 7: Rotate gripper back to 0 degrees (for picking tomatoes)
        print("[Step 7] Rotate gripper to 0 degrees")
        # Rotate back to original orientation
        obs_now = task.get_observation()
        current_quat = obs_now.gripper_pose[3:7]
        r = R.from_quat(current_quat)
        r_0 = r * R.from_euler('z', -90, degrees=True)
        target_quat_0 = r_0.as_quat()
        obs, reward, done = rotate(env, task, target_quat_0)
        if done:
            print("[Task] Task ended at step 7!")
            return

        # Step 8: Move to item1 (first tomato)
        print("[Step 8] Move to item1")
        obs, reward, done = move(env, task, item1_pos)
        if done:
            print("[Task] Task ended at step 8!")
            return

        # Step 9: Pick item1
        print("[Step 9] Pick item1")
        obs, reward, done = pick(env, task, item1_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended at step 9!")
            return

        # Step 10: Move to plate
        print("[Step 10] Move to plate")
        obs, reward, done = move(env, task, plate_pos)
        if done:
            print("[Task] Task ended at step 10!")
            return

        # Step 11: Place item1 on plate
        print("[Step 11] Place item1 on plate")
        obs, reward, done = place(env, task, plate_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended at step 11!")
            return

        # Step 12: Move to item2
        print("[Step 12] Move to item2")
        obs, reward, done = move(env, task, item2_pos)
        if done:
            print("[Task] Task ended at step 12!")
            return

        # Step 13: Pick item2
        print("[Step 13] Pick item2")
        obs, reward, done = pick(env, task, item2_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended at step 13!")
            return

        # Step 14: Move to plate
        print("[Step 14] Move to plate")
        obs, reward, done = move(env, task, plate_pos)
        if done:
            print("[Task] Task ended at step 14!")
            return

        # Step 15: Place item2 on plate
        print("[Step 15] Place item2 on plate")
        obs, reward, done = place(env, task, plate_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended at step 15!")
            return

        # Optionally, repeat for item3 if required by the goal (the goal says "all the tomatoes")
        # If item3 is a tomato, repeat the pick and place for item3
        if item3_pos is not None:
            print("[Step 16] Move to item3")
            obs, reward, done = move(env, task, item3_pos)
            if done:
                print("[Task] Task ended at step 16!")
                return

            print("[Step 17] Pick item3")
            obs, reward, done = pick(env, task, item3_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended at step 17!")
                return

            print("[Step 18] Move to plate")
            obs, reward, done = move(env, task, plate_pos)
            if done:
                print("[Task] Task ended at step 18!")
                return

            print("[Step 19] Place item3 on plate")
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended at step 19!")
                return

        print("[Task] Task completed successfully! Reward:", reward)

    except Exception as e:
        print("[Task] Exception occurred:", e)
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()