import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task logic for: Unlock the cabinet by opening an accessible drawer, then transfer the 2 tomatoes onto the plate, leaving any other items undisturbed.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object name mapping (from object list and likely RLBench conventions)
        # 'item1', 'item2' are the tomatoes; 'item3' is the rubbish; 'plate' is the goal location
        # Drawer positions: 'bottom_anchor_pos', 'bottom_side_pos', etc.

        # --- Step 1: Move gripper to the side position of the bottom drawer ---
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] Could not find 'bottom_side_pos' in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Step 1: Move to bottom drawer side position.")
        obs, reward, done = move(
            env,
            task,
            target_pos=bottom_side_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # --- Step 2: Rotate gripper to 90 degrees (for drawer grasp) ---
        # We'll assume the target orientation for 90 degrees is provided by the object position system
        # or can be constructed as a quaternion. Let's try to get it from the drawer joint or use a default.
        try:
            # If positions dict provides orientation, use it; else, construct 90deg rotation about z
            # For RLBench, gripper orientation is usually [x, y, z, w] quaternion
            # We'll rotate about z by 90 degrees
            from scipy.spatial.transform import Rotation as R
            ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        except Exception as e:
            print("[Error] Could not compute 90 degree quaternion:", e)
            shutdown_environment(env)
            return

        print("[Plan] Step 2: Rotate gripper to 90 degrees.")
        obs, reward, done = rotate(
            env,
            task,
            target_quat=ninety_deg_quat,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after rotate!")
            return

        # --- Step 3: Pull the drawer open ---
        # We'll pull along the x-axis (assume drawer opens along +x, adjust if needed)
        pull_distance = 0.15  # 15cm, adjust as needed for your environment
        print("[Plan] Step 3: Pull the drawer open.")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=pull_distance,
            pull_axis='x',
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pull!")
            return

        # --- Step 4: Move to item1 (tomato1) ---
        try:
            item1_pos = positions['item1']
        except KeyError:
            print("[Error] Could not find 'item1' in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Step 4: Move to item1 (tomato1).")
        obs, reward, done = move(
            env,
            task,
            target_pos=item1_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to item1!")
            return

        # --- Step 5: Pick item1 (tomato1) ---
        print("[Plan] Step 5: Pick item1 (tomato1).")
        obs, reward, done = pick(
            env,
            task,
            target_pos=item1_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick item1!")
            return

        # --- Step 6: Place item1 on plate ---
        try:
            plate_pos = positions['plate']
        except KeyError:
            print("[Error] Could not find 'plate' in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Step 6: Place item1 on plate.")
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after place item1!")
            return

        # --- Step 7: Move to item2 (tomato2) ---
        try:
            item2_pos = positions['item2']
        except KeyError:
            print("[Error] Could not find 'item2' in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Step 7: Move to item2 (tomato2).")
        obs, reward, done = move(
            env,
            task,
            target_pos=item2_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to item2!")
            return

        # --- Step 8: Pick item2 (tomato2) ---
        print("[Plan] Step 8: Pick item2 (tomato2).")
        obs, reward, done = pick(
            env,
            task,
            target_pos=item2_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick item2!")
            return

        # --- Step 9: Place item2 on plate ---
        print("[Plan] Step 9: Place item2 on plate.")
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after place item2!")
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()