import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object keys: 'item1', 'item2', 'item3', 'plate', plus drawer positions

        # For this task, "tomato1" and "tomato2" are not in the object list, but "item1" and "item2" are.
        # We'll treat item1 and item2 as the two tomatoes.

        # 1. Slide open any available drawer to access its contents.
        # We'll use the "bottom" drawer for this example.

        # Drawer positions
        bottom_side_pos = positions.get('bottom_side_pos', None)
        bottom_anchor_pos = positions.get('bottom_anchor_pos', None)
        bottom_joint_pos = positions.get('bottom_joint_pos', None)
        # Gripper starts at 'nowhere-pos', but we use the actual gripper pose from obs

        # 1. Move to the side position of the bottom drawer
        if bottom_side_pos is None:
            print("[Error] bottom_side_pos not found in object positions.")
            return
        print("[Plan] Moving to bottom_side_pos:", bottom_side_pos)
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # 2. Rotate gripper to 90 degrees (for drawer grasp)
        # We'll use quaternion for 90 deg rotation about z axis
        # Get current gripper pose
        obs = task.get_observation()
        current_quat = obs.gripper_pose[3:7]
        # 90 deg about z axis: quaternion [x, y, z, w] = [0, 0, sqrt(0.5), sqrt(0.5)]
        target_quat = np.array([0, 0, np.sqrt(0.5), np.sqrt(0.5)])
        print("[Plan] Rotating gripper to 90 degrees (about z axis)")
        obs, reward, done = rotate(env, task, target_quat=target_quat)
        if done:
            print("[Task] Task ended after rotate!")
            return

        # 3. Move to anchor position of the bottom drawer (for grasping handle)
        if bottom_anchor_pos is None:
            print("[Error] bottom_anchor_pos not found in object positions.")
            return
        print("[Plan] Moving to bottom_anchor_pos:", bottom_anchor_pos)
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # 4. Pull the drawer open (along x axis, positive direction)
        # We'll assume a pull distance of 0.15 meters
        pull_distance = 0.15
        print("[Plan] Pulling the drawer open along +x axis, distance:", pull_distance)
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Task] Task ended after pull!")
            return

        # 5. Move to the table to pick up the first tomato (item1)
        item1_pos = positions.get('item1', None)
        if item1_pos is None:
            print("[Error] item1 not found in object positions.")
            return
        print("[Plan] Moving to item1 (tomato1) at:", item1_pos)
        obs, reward, done = move(env, task, target_pos=item1_pos)
        if done:
            print("[Task] Task ended after move to item1!")
            return

        # 6. Pick up item1 (tomato1)
        print("[Plan] Picking up item1 (tomato1)")
        obs, reward, done = pick(env, task, target_pos=item1_pos)
        if done:
            print("[Task] Task ended after pick item1!")
            return

        # 7. Move to the plate to place the tomato
        plate_pos = positions.get('plate', None)
        if plate_pos is None:
            print("[Error] plate not found in object positions.")
            return
        print("[Plan] Moving to plate at:", plate_pos)
        obs, reward, done = move(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Task ended after move to plate!")
            return

        # 8. Place item1 (tomato1) on the plate
        print("[Plan] Placing item1 (tomato1) on the plate")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Task ended after place item1!")
            return

        # 9. Move to the table to pick up the second tomato (item2)
        item2_pos = positions.get('item2', None)
        if item2_pos is None:
            print("[Error] item2 not found in object positions.")
            return
        print("[Plan] Moving to item2 (tomato2) at:", item2_pos)
        obs, reward, done = move(env, task, target_pos=item2_pos)
        if done:
            print("[Task] Task ended after move to item2!")
            return

        # 10. Pick up item2 (tomato2)
        print("[Plan] Picking up item2 (tomato2)")
        obs, reward, done = pick(env, task, target_pos=item2_pos)
        if done:
            print("[Task] Task ended after pick item2!")
            return

        # 11. Move to the plate to place the second tomato
        print("[Plan] Moving to plate at:", plate_pos)
        obs, reward, done = move(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Task ended after move to plate (second tomato)!")
            return

        # 12. Place item2 (tomato2) on the plate
        print("[Plan] Placing item2 (tomato2) on the plate")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Task ended after place item2!")
            return

        print("[Task] Task completed successfully! Reward:", reward)

    except Exception as e:
        print("[Exception] An error occurred during task execution:", str(e))
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()