import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list:
        # [
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos',
        # 'item1', 'item2', 'item3', 'plate'
        # ]

        # For the goal: "Open one of the drawers in the cabinet, then put all tomatoes on the plate."
        # We will:
        # 1. Move to the side position of a drawer (let's use 'bottom' drawer for this plan)
        # 2. Rotate gripper to 90 degrees (for side grasp)
        # 3. Move to anchor position of the drawer
        # 4. Pull the drawer open
        # 5. Move to the first tomato (item1), pick it, place on plate
        # 6. Repeat for item2 and item3

        # Defensive: Check all required keys exist
        required_keys = [
            'bottom_side_pos', 'bottom_anchor_pos', 'item1', 'item2', 'item3', 'plate'
        ]
        for k in required_keys:
            if k not in positions:
                raise RuntimeError(f"Missing required object position: {k}")

        # Step 1: Move to the side position of the bottom drawer
        print("[Plan] Step 1: Move to bottom_side_pos")
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(positions['bottom_side_pos']),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # Step 2: Rotate gripper to 90 degrees (for side grasp)
        print("[Plan] Step 2: Rotate gripper to 90 degrees")
        # We'll assume the quaternion for 90 deg rotation about z axis
        # If the gripper's default orientation is aligned with zero_deg, then a 90 deg rotation about z:
        # Use scipy to get quaternion for [0, 0, np.pi/2] euler
        from scipy.spatial.transform import Rotation as R
        target_euler = [0, 0, np.pi/2]
        target_quat = R.from_euler('xyz', target_euler).as_quat()  # xyzw
        obs, reward, done = rotate(
            env,
            task,
            target_quat=target_quat,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after rotate!")
            return

        # Step 3: Move to anchor position of the bottom drawer
        print("[Plan] Step 3: Move to bottom_anchor_pos")
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(positions['bottom_anchor_pos']),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # Step 4: Pull the drawer open (pull along x axis, positive direction)
        print("[Plan] Step 4: Pull the drawer open")
        # We'll use a reasonable pull distance (e.g., 0.15m)
        pull_distance = 0.15
        obs, reward, done = pull(
            env,
            task,
            pull_distance=pull_distance,
            pull_axis='x',
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pull!")
            return

        # Step 5: Move to item1, pick, place on plate
        print("[Plan] Step 5: Move to item1")
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(positions['item1']),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to item1!")
            return

        print("[Plan] Step 6: Pick item1")
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(positions['item1']),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick item1!")
            return

        print("[Plan] Step 7: Place item1 on plate")
        obs, reward, done = place(
            env,
            task,
            target_pos=np.array(positions['plate']),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after place item1!")
            return

        # Step 8: Move to item2, pick, place on plate
        print("[Plan] Step 8: Move to item2")
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(positions['item2']),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to item2!")
            return

        print("[Plan] Step 9: Pick item2")
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(positions['item2']),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick item2!")
            return

        print("[Plan] Step 10: Place item2 on plate")
        obs, reward, done = place(
            env,
            task,
            target_pos=np.array(positions['plate']),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after place item2!")
            return

        # Step 11: Move to item3, pick, place on plate
        print("[Plan] Step 11: Move to item3")
        obs, reward, done = move(
            env,
            task,
            target_pos=np.array(positions['item3']),
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to item3!")
            return

        print("[Plan] Step 12: Pick item3")
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(positions['item3']),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick item3!")
            return

        print("[Plan] Step 13: Place item3 on plate")
        obs, reward, done = place(
            env,
            task,
            target_pos=np.array(positions['plate']),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after place item3!")
            return

        print("[Task] All tomatoes placed on plate and drawer opened. Task completed successfully!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()