import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object keys: 'item1', 'item2', 'item3', 'plate', 'bottom_anchor_pos', 'bottom_side_pos', etc.

        # --- PLAN EXECUTION ---
        # Goal: Slide open any available drawer to access its contents, then toss the 2 tomatoes from the table onto the plate.
        # We will use the 'bottom' drawer (bottom_anchor_pos, bottom_side_pos).
        # We'll treat 'item1' and 'item2' as the tomatoes.

        # Step 1: Move to the side position of the bottom drawer
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] bottom_side_pos not found in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Moving to bottom_side_pos:", bottom_side_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=bottom_side_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # Step 2: Rotate gripper to 90 degrees (for drawer grasp)
        # We'll assume the gripper's orientation is quaternion, and 90 deg about z axis.
        # We'll use scipy's Rotation to get the quaternion for 90 deg about z.
        from scipy.spatial.transform import Rotation as R
        quat_ninety_deg = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        print("[Plan] Rotating gripper to 90 degrees (about z axis):", quat_ninety_deg)
        obs, reward, done = rotate(
            env,
            task,
            target_quat=quat_ninety_deg,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after rotate!")
            return

        # Step 3: Move to the anchor position of the bottom drawer (for grasping)
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Error] bottom_anchor_pos not found in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Moving to bottom_anchor_pos:", bottom_anchor_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=bottom_anchor_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # Step 4: Pull the drawer open (along x axis, e.g. +0.15m)
        # We'll use a pull distance of 0.15m along the x axis.
        pull_distance = 0.15
        print("[Plan] Pulling drawer along +x by", pull_distance)
        obs, reward, done = pull(
            env,
            task,
            pull_distance=pull_distance,
            pull_axis='x',
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pull!")
            return

        # Step 5: Move to the table position to pick up the first tomato (item1)
        try:
            item1_pos = positions['item1']
        except KeyError:
            print("[Error] item1 not found in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Moving to item1 (tomato1) at:", item1_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=item1_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to item1!")
            return

        # Step 6: Pick up item1 (tomato1)
        print("[Plan] Picking up item1 (tomato1)")
        obs, reward, done = pick(
            env,
            task,
            target_pos=item1_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick item1!")
            return

        # Step 7: Move to the plate to place the tomato
        try:
            plate_pos = positions['plate']
        except KeyError:
            print("[Error] plate not found in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Moving to plate at:", plate_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=plate_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to plate (for item1)!")
            return

        # Step 8: Place item1 (tomato1) on the plate
        print("[Plan] Placing item1 (tomato1) on plate")
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after place item1!")
            return

        # Step 9: Move to the table position to pick up the second tomato (item2)
        try:
            item2_pos = positions['item2']
        except KeyError:
            print("[Error] item2 not found in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Moving to item2 (tomato2) at:", item2_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=item2_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to item2!")
            return

        # Step 10: Pick up item2 (tomato2)
        print("[Plan] Picking up item2 (tomato2)")
        obs, reward, done = pick(
            env,
            task,
            target_pos=item2_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick item2!")
            return

        # Step 11: Move to the plate to place the second tomato
        print("[Plan] Moving to plate at:", plate_pos)
        obs, reward, done = move(
            env,
            task,
            target_pos=plate_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to plate (for item2)!")
            return

        # Step 12: Place item2 (tomato2) on the plate
        print("[Plan] Placing item2 (tomato2) on plate")
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after place item2!")
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()