import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Task Plan Execution ===
        # Goal: Pull open any drawer that is not locked, then drop the 2 tomatoes onto the plate.

        # 1. Choose a drawer to open (bottom, middle, or top). All are present; assume none are locked.
        # We'll use the 'bottom' drawer for this plan.

        # Get relevant positions
        try:
            gripper_pos = positions['waypoint1'] if 'waypoint1' in positions else positions['bottom_side_pos']
            bottom_side_pos = positions['bottom_side_pos']
            bottom_anchor_pos = positions['bottom_anchor_pos']
            bottom_joint_pos = positions['bottom_joint_pos']
            tomato1_pos = positions['tomato1']
            tomato2_pos = positions['tomato2']
            plate_pos = positions['plate']
        except KeyError as e:
            print(f"[Error] Missing object position for: {e}")
            shutdown_environment(env)
            return

        # Step 1: Rotate gripper to 90 degrees (for side approach)
        # We'll use quaternion for 90 deg rotation about z axis
        # RLBench uses [x, y, z, w] format for quaternions
        from scipy.spatial.transform import Rotation as R
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # [x, y, z, w]
        obs, reward, done = rotate(env, task, ninety_deg_quat)
        if done:
            print("[Task] Task ended after rotate!")
            return

        # Step 2: Move to the side position of the bottom drawer
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Task ended after move to side position!")
            return

        # Step 3: Move to the anchor position of the bottom drawer (for grasping handle)
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Task ended after move to anchor position!")
            return

        # Step 4: Pick the drawer handle (simulate grasping the handle)
        # For the drawer, we use the anchor position as the pick target
        obs, reward, done = pick(env, task, bottom_anchor_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended after picking drawer handle!")
            return

        # Step 5: Pull the drawer open (pull along x axis, positive direction)
        # Assume pulling 0.15 meters is sufficient to open
        obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
        if done:
            print("[Task] Task ended after pulling drawer!")
            return

        # Step 6: Pick up tomato1 from the table
        obs, reward, done = pick(env, task, tomato1_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after picking tomato1!")
            return

        # Step 7: Place tomato1 onto the plate
        obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after placing tomato1!")
            return

        # Step 8: Pick up tomato2 from the table
        obs, reward, done = pick(env, task, tomato2_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after picking tomato2!")
            return

        # Step 9: Place tomato2 onto the plate
        obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    except Exception as e:
        print(f"[Exception] {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()