import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer fully, then pick up all tomatoes and place them on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object keys expected in positions: 
        # 'middle_side_pos', 'middle_anchor_pos', 'tomato1', 'tomato2', 'plate', etc.

        # --- Step 1: Rotate gripper to ninety_deg orientation ---
        # We'll assume 'ninety_deg' means a quaternion corresponding to 90 deg rotation about z.
        # If positions contains 'gripper_ninety_deg_quat', use it; else, compute it.
        try:
            # Try to get the quaternion for 90 deg rotation about z axis
            from scipy.spatial.transform import Rotation as R
            ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        except Exception as e:
            print("[Task] Could not compute 90 deg quaternion, using default [0,0,0,1]")
            ninety_deg_quat = np.array([0, 0, 0, 1])

        print("[Task] Step 1: Rotate gripper to 90 degrees (about z axis)")
        try:
            obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Task] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # --- Step 2: Move gripper to side position of middle drawer ---
        print("[Task] Step 2: Move gripper to side-pos-middle")
        try:
            side_pos = positions.get('middle_side_pos', None)
            if side_pos is None:
                raise KeyError("middle_side_pos not found in object positions.")
            obs, reward, done = move(env, task, target_pos=np.array(side_pos))
            if done:
                print("[Task] Task ended after move to side-pos-middle!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to side-pos-middle: {e}")
            shutdown_environment(env)
            return

        # --- Step 3: Move gripper to anchor position of middle drawer ---
        print("[Task] Step 3: Move gripper to anchor-pos-middle")
        try:
            anchor_pos = positions.get('middle_anchor_pos', None)
            if anchor_pos is None:
                raise KeyError("middle_anchor_pos not found in object positions.")
            obs, reward, done = move(env, task, target_pos=np.array(anchor_pos))
            if done:
                print("[Task] Task ended after move to anchor-pos-middle!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to anchor-pos-middle: {e}")
            shutdown_environment(env)
            return

        # --- Step 4: Pick the middle drawer handle (simulate pick-drawer) ---
        print("[Task] Step 4: Pick (grasp) the middle drawer at anchor-pos-middle")
        try:
            # We use pick skill to grasp the drawer handle at anchor_pos
            obs, reward, done = pick(env, task, target_pos=np.array(anchor_pos), approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after pick-drawer!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick-drawer: {e}")
            shutdown_environment(env)
            return

        # --- Step 5: Pull the drawer open (simulate pull) ---
        print("[Task] Step 5: Pull the middle drawer open")
        try:
            # Pull along x axis (assume drawer opens along +x, adjust if needed)
            pull_distance = 0.18  # Adjust as needed for "fully open"
            obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[Task] Exception during pull: {e}")
            shutdown_environment(env)
            return

        # --- Step 6: Pick up tomato1 from table ---
        print("[Task] Step 6: Pick up tomato1 from table")
        try:
            tomato1_pos = positions.get('tomato1', None)
            if tomato1_pos is None:
                raise KeyError("tomato1 not found in object positions.")
            obs, reward, done = pick(env, task, target_pos=np.array(tomato1_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after picking tomato1!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick tomato1: {e}")
            shutdown_environment(env)
            return

        # --- Step 7: Place tomato1 on plate ---
        print("[Task] Step 7: Place tomato1 on plate")
        try:
            plate_pos = positions.get('plate', None)
            if plate_pos is None:
                raise KeyError("plate not found in object positions.")
            obs, reward, done = place(env, task, target_pos=np.array(plate_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after placing tomato1!")
                return
        except Exception as e:
            print(f"[Task] Exception during place tomato1: {e}")
            shutdown_environment(env)
            return

        # --- Step 8: Pick up tomato2 from table ---
        print("[Task] Step 8: Pick up tomato2 from table")
        try:
            tomato2_pos = positions.get('tomato2', None)
            if tomato2_pos is None:
                raise KeyError("tomato2 not found in object positions.")
            obs, reward, done = pick(env, task, target_pos=np.array(tomato2_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after picking tomato2!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick tomato2: {e}")
            shutdown_environment(env)
            return

        # --- Step 9: Place tomato2 on plate ---
        print("[Task] Step 9: Place tomato2 on plate")
        try:
            obs, reward, done = place(env, task, target_pos=np.array(plate_pos), approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after placing tomato2!")
                return
        except Exception as e:
            print(f"[Task] Exception during place tomato2: {e}")
            shutdown_environment(env)
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()