# run_skeleton_task.py (Completed Executable Code)

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Map object names to their positions
        # The object list includes: 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos', 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos', 'top_anchor_pos', 'top_joint_pos', 'top_side_pos', 'waypoint1', 'rubbish', 'tomato1', 'tomato2', 'plate'
        # We'll use: 'bottom_side_pos', 'bottom_anchor_pos', 'tomato1', 'tomato2', 'plate'
        # Defensive: check for presence
        required_keys = ['bottom_side_pos', 'bottom_anchor_pos', 'tomato1', 'tomato2', 'plate']
        for k in required_keys:
            if k not in positions:
                raise RuntimeError(f"Required object '{k}' not found in object_positions.")

        bottom_side_pos = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']
        plate_pos = positions['plate']

        # === Plan Execution ===
        # Plan steps (from specification):
        # 1. rotate(gripper, zero_deg, ninety_deg)
        # 2. move(gripper, side-pos-bottom)
        # 3. move(gripper, anchor-pos-bottom)
        # 4. (pick-drawer, not available, skip)
        # 5. pull(gripper, bottom)
        # 6. pick(tomato1, table)
        # 7. place(tomato1, plate)
        # 8. pick(tomato2, table)
        # 9. place(tomato2, plate)

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        # We'll assume zero_deg = [0,0,0,1] and ninety_deg = [0,0,sqrt(0.5),sqrt(0.5)] in quaternion (xyzw)
        # But let's try to get the current orientation and rotate 90deg about z
        try:
            obs = task.get_observation()
            start_quat = obs.gripper_pose[3:7]
            # Compose a 90deg rotation about z
            from scipy.spatial.transform import Rotation as R
            r = R.from_quat(start_quat)
            r90 = R.from_euler('z', 90, degrees=True)
            target_quat = (r * r90).as_quat()
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[Task] Exception during rotate: {e}")
            shutdown_environment(env)
            raise

        # Step 2: Move gripper to side position of bottom drawer
        try:
            obs, reward, done = move(env, task, bottom_side_pos)
            if done:
                print("[Task] Task ended after move to side-pos-bottom!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to side-pos-bottom: {e}")
            shutdown_environment(env)
            raise

        # Step 3: Move gripper to anchor position of bottom drawer
        try:
            obs, reward, done = move(env, task, bottom_anchor_pos)
            if done:
                print("[Task] Task ended after move to anchor-pos-bottom!")
                return
        except Exception as e:
            print(f"[Task] Exception during move to anchor-pos-bottom: {e}")
            shutdown_environment(env)
            raise

        # Step 4: pick-drawer (not available, skip)
        print("[Task] Skipping pick-drawer step (not available as a skill).")

        # Step 5: Pull the drawer open
        # We need to pull along the x axis (assume positive x is outwards), and estimate a reasonable pull distance
        # Let's try pulling 0.18 meters along x
        try:
            pull_distance = 0.18
            obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[Task] Exception during pull: {e}")
            shutdown_environment(env)
            raise

        # Step 6: Pick up tomato1 from table (use tomato1_pos)
        try:
            obs, reward, done = pick(env, task, tomato1_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick tomato1!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick tomato1: {e}")
            shutdown_environment(env)
            raise

        # Step 7: Place tomato1 on plate
        try:
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after place tomato1!")
                return
        except Exception as e:
            print(f"[Task] Exception during place tomato1: {e}")
            shutdown_environment(env)
            raise

        # Step 8: Pick up tomato2 from table (use tomato2_pos)
        try:
            obs, reward, done = pick(env, task, tomato2_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick tomato2!")
                return
        except Exception as e:
            print(f"[Task] Exception during pick tomato2: {e}")
            shutdown_environment(env)
            raise

        # Step 9: Place tomato2 on plate
        try:
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after place tomato2!")
                return
        except Exception as e:
            print(f"[Task] Exception during place tomato2: {e}")
            shutdown_environment(env)
            raise

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()