import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task logic for: Unlock the cabinet by opening an accessible drawer, then transfer the 2 tomatoes onto the plate, leaving any other items undisturbed.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and environment:
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos',
        # 'waypoint1', 'rubbish', 'tomato1', 'tomato2', 'plate'

        # For this plan, we will open the bottom drawer (as it is likely accessible)
        # and then move both tomatoes to the plate.

        # --- Step 1: Rotate gripper to 90 degrees (vertical for drawer manipulation) ---
        try:
            # Get current gripper pose
            obs = task.get_observation()
            # Target orientation: 90 degrees about z (assuming 'ninety_deg' means vertical)
            # We'll use the orientation of the 'bottom_anchor_pos' as a reference if available
            # Otherwise, rotate 90 deg about z from current orientation
            # For RLBench, let's assume 'ninety_deg' means [0, 0, np.pi/2] in euler
            from scipy.spatial.transform import Rotation as R
            target_euler = [0, 0, np.pi/2]
            target_quat = R.from_euler('xyz', target_euler).as_quat()  # xyzw
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[ERROR] Step 1 (rotate) failed: {e}")
            return

        # --- Step 2: Move gripper to the side position of the bottom drawer ---
        try:
            bottom_side_pos = positions.get('bottom_side_pos')
            if bottom_side_pos is None:
                raise KeyError("bottom_side_pos not found in positions.")
            obs, reward, done = move(env, task, bottom_side_pos)
            if done:
                print("[Task] Task ended after move to bottom_side_pos!")
                return
        except Exception as e:
            print(f"[ERROR] Step 2 (move to bottom_side_pos) failed: {e}")
            return

        # --- Step 3: Move gripper to the anchor position of the bottom drawer (for grasping handle) ---
        try:
            bottom_anchor_pos = positions.get('bottom_anchor_pos')
            if bottom_anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in positions.")
            obs, reward, done = move(env, task, bottom_anchor_pos)
            if done:
                print("[Task] Task ended after move to bottom_anchor_pos!")
                return
        except Exception as e:
            print(f"[ERROR] Step 3 (move to bottom_anchor_pos) failed: {e}")
            return

        # --- Step 4: Pick the drawer handle (simulate grasping the drawer) ---
        try:
            # Use pick at the anchor position (simulate grasping the handle)
            obs, reward, done = pick(env, task, bottom_anchor_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended after pick drawer handle!")
                return
        except Exception as e:
            print(f"[ERROR] Step 4 (pick drawer handle) failed: {e}")
            return

        # --- Step 5: Pull the drawer open (pull along x axis, e.g., 0.15m) ---
        try:
            # Pull distance: 0.15m (adjust as needed for your environment)
            obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
            if done:
                print("[Task] Task ended after pull (open drawer)!")
                return
        except Exception as e:
            print(f"[ERROR] Step 5 (pull drawer) failed: {e}")
            return

        # --- Step 6: Pick up tomato1 from the table ---
        try:
            tomato1_pos = positions.get('tomato1')
            if tomato1_pos is None:
                raise KeyError("tomato1 not found in positions.")
            obs, reward, done = pick(env, task, tomato1_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] Step 6 (pick tomato1) failed: {e}")
            return

        # --- Step 7: Place tomato1 on the plate ---
        try:
            plate_pos = positions.get('plate')
            if plate_pos is None:
                raise KeyError("plate not found in positions.")
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after place tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] Step 7 (place tomato1) failed: {e}")
            return

        # --- Step 8: Pick up tomato2 from the table ---
        try:
            tomato2_pos = positions.get('tomato2')
            if tomato2_pos is None:
                raise KeyError("tomato2 not found in positions.")
            obs, reward, done = pick(env, task, tomato2_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended after pick tomato2!")
                return
        except Exception as e:
            print(f"[ERROR] Step 8 (pick tomato2) failed: {e}")
            return

        # --- Step 9: Place tomato2 on the plate ---
        try:
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[ERROR] Step 9 (place tomato2) failed: {e}")
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()