import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer fully, then pick up all the tomatoes and leave them on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list
        # 'item1', 'item2', 'item3' are the tomatoes
        # 'plate' is the goal location
        # Drawer positions: 'bottom_anchor_pos', 'bottom_side_pos', etc.

        # For robustness, check all required keys
        required_keys = [
            'bottom_anchor_pos', 'bottom_side_pos', 'bottom_joint_pos',
            'middle_anchor_pos', 'middle_side_pos', 'middle_joint_pos',
            'top_anchor_pos', 'top_side_pos', 'top_joint_pos',
            'item1', 'item2', 'item3', 'plate'
        ]
        for k in required_keys:
            if k not in positions:
                raise RuntimeError(f"Missing object position for: {k}")

        # Choose a drawer to open (let's use the 'bottom' drawer)
        drawer_anchor = positions['bottom_anchor_pos']
        drawer_side = positions['bottom_side_pos']
        drawer_joint = positions['bottom_joint_pos']
        # For pulling, we will pull along the x-axis (assume drawer opens along +x)
        pull_axis = 'x'
        pull_distance = 0.18  # Empirically chosen; adjust as needed for full open

        # 1. Move to the side position of the drawer (move gripper above side handle)
        print("[Plan] Step 1: Move to side position of bottom drawer.")
        obs, reward, done = move(env, task, target_pos=drawer_side)
        if done:
            print("[Task] Task ended after move to side position!")
            return

        # 2. Rotate gripper to 90 degrees (assume this is required for grasping the handle)
        print("[Plan] Step 2: Rotate gripper to 90 degrees for handle grasp.")
        # Get current gripper position and orientation
        obs_now = task.get_observation()
        # Convert 90 degrees about z axis to quaternion
        from scipy.spatial.transform import Rotation as R
        quat_90deg = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        obs, reward, done = rotate(env, task, target_quat=quat_90deg)
        if done:
            print("[Task] Task ended after rotate!")
            return

        # 3. Move to anchor position (where the handle can be grasped)
        print("[Plan] Step 3: Move to anchor position of bottom drawer.")
        obs, reward, done = move(env, task, target_pos=drawer_anchor)
        if done:
            print("[Task] Task ended after move to anchor position!")
            return

        # 4. Pick the drawer handle (simulate grasping the handle)
        print("[Plan] Step 4: Pick (grasp) the drawer handle at anchor position.")
        # Use pick skill to close gripper at anchor position
        obs, reward, done = pick(env, task, target_pos=drawer_anchor, approach_distance=0.08, approach_axis='z')
        if done:
            print("[Task] Task ended after picking drawer handle!")
            return

        # 5. Pull the drawer open (along x axis)
        print("[Plan] Step 5: Pull the drawer open.")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis=pull_axis)
        if done:
            print("[Task] Task ended after pulling drawer!")
            return

        # 6. Move away from the drawer (to avoid collision, move back to side position)
        print("[Plan] Step 6: Move back to side position of bottom drawer.")
        obs, reward, done = move(env, task, target_pos=drawer_side)
        if done:
            print("[Task] Task ended after moving away from drawer!")
            return

        # 7. Rotate gripper back to 0 degrees (for picking tomatoes)
        print("[Plan] Step 7: Rotate gripper back to 0 degrees.")
        quat_0deg = R.from_euler('z', 0, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat=quat_0deg)
        if done:
            print("[Task] Task ended after rotate to 0 deg!")
            return

        # 8. Move to above first tomato (item1)
        print("[Plan] Step 8: Move to item1 (first tomato).")
        obs, reward, done = move(env, task, target_pos=positions['item1'])
        if done:
            print("[Task] Task ended after moving to item1!")
            return

        # 9. Pick item1
        print("[Plan] Step 9: Pick item1.")
        obs, reward, done = pick(env, task, target_pos=positions['item1'], approach_distance=0.12, approach_axis='z')
        if done:
            print("[Task] Task ended after picking item1!")
            return

        # 10. Place item1 on plate
        print("[Plan] Step 10: Place item1 on plate.")
        obs, reward, done = place(env, task, target_pos=positions['plate'], approach_distance=0.12, approach_axis='z')
        if done:
            print("[Task] Task ended after placing item1!")
            return

        # 11. Move to item2
        print("[Plan] Step 11: Move to item2 (second tomato).")
        obs, reward, done = move(env, task, target_pos=positions['item2'])
        if done:
            print("[Task] Task ended after moving to item2!")
            return

        # 12. Pick item2
        print("[Plan] Step 12: Pick item2.")
        obs, reward, done = pick(env, task, target_pos=positions['item2'], approach_distance=0.12, approach_axis='z')
        if done:
            print("[Task] Task ended after picking item2!")
            return

        # 13. Place item2 on plate
        print("[Plan] Step 13: Place item2 on plate.")
        obs, reward, done = place(env, task, target_pos=positions['plate'], approach_distance=0.12, approach_axis='z')
        if done:
            print("[Task] Task ended after placing item2!")
            return

        # 14. Move to item3
        print("[Plan] Step 14: Move to item3 (third tomato).")
        obs, reward, done = move(env, task, target_pos=positions['item3'])
        if done:
            print("[Task] Task ended after moving to item3!")
            return

        # 15. Pick item3
        print("[Plan] Step 15: Pick item3.")
        obs, reward, done = pick(env, task, target_pos=positions['item3'], approach_distance=0.12, approach_axis='z')
        if done:
            print("[Task] Task ended after picking item3!")
            return

        # 16. Place item3 on plate
        print("[Plan] Step 16: Place item3 on plate.")
        obs, reward, done = place(env, task, target_pos=positions['plate'], approach_distance=0.12, approach_axis='z')
        if done:
            print("[Task] Task ended after placing item3!")
            return

        print("[Task] All tomatoes placed on plate. Task completed successfully!")

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()