# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Task: Open an accessible drawer to unlock the cabinet, then transfer the 2 tomatoes onto the plate, leaving other items undisturbed.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and observation:
        # 'item1', 'item2', 'item3', 'plate'
        # Drawer positions: 'bottom_anchor_pos', 'bottom_side_pos', etc.

        # --- Step 1: Move to the side position of the bottom drawer ---
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] Could not find 'bottom_side_pos' in object positions.")
            shutdown_environment(env)
            return
        print("[Plan] Step 1: Move to bottom_side_pos (side of bottom drawer)")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # --- Step 2: Rotate gripper to 90 degrees (for drawer handle alignment) ---
        # Assume 90 deg = [x, y, z, w] quaternion for 90 deg about z axis
        # We'll use scipy to get the quaternion for 90 deg about z
        from scipy.spatial.transform import Rotation as R
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        print("[Plan] Step 2: Rotate gripper to 90 degrees")
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        if done:
            print("[Task] Task ended after rotate!")
            return

        # --- Step 3: Pull the drawer open ---
        # Move to anchor position (handle), then pull along x axis
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Error] Could not find 'bottom_anchor_pos' in object positions.")
            shutdown_environment(env)
            return
        print("[Plan] Step 3a: Move to bottom_anchor_pos (anchor/handle of bottom drawer)")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # Pull: estimate pull distance (e.g., 0.15m along x axis)
        pull_distance = 0.15
        print("[Plan] Step 3b: Pull the drawer open along x axis")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Task] Task ended after pull!")
            return

        # --- Step 4: Move above item1 (tomato1) on the table ---
        try:
            item1_pos = positions['item1']
        except KeyError:
            print("[Error] Could not find 'item1' in object positions.")
            shutdown_environment(env)
            return
        # Approach above item1 (z+0.10)
        approach_item1_pos = np.array(item1_pos) + np.array([0, 0, 0.10])
        print("[Plan] Step 4: Move above item1 (tomato1)")
        obs, reward, done = move(env, task, target_pos=approach_item1_pos)
        if done:
            print("[Task] Task ended after move above item1!")
            return

        # --- Step 5: Pick item1 (tomato1) ---
        print("[Plan] Step 5: Pick item1 (tomato1)")
        obs, reward, done = pick(env, task, target_pos=np.array(item1_pos), approach_distance=0.10, approach_axis='-z')
        if done:
            print("[Task] Task ended after pick item1!")
            return

        # --- Step 6: Place item1 on the plate ---
        try:
            plate_pos = positions['plate']
        except KeyError:
            print("[Error] Could not find 'plate' in object positions.")
            shutdown_environment(env)
            return
        # Approach above plate (z+0.10)
        approach_plate_pos = np.array(plate_pos) + np.array([0, 0, 0.10])
        print("[Plan] Step 6a: Move above plate")
        obs, reward, done = move(env, task, target_pos=approach_plate_pos)
        if done:
            print("[Task] Task ended after move above plate!")
            return
        print("[Plan] Step 6b: Place item1 on plate")
        obs, reward, done = place(env, task, target_pos=np.array(plate_pos), approach_distance=0.10, approach_axis='-z')
        if done:
            print("[Task] Task ended after place item1!")
            return

        # --- Step 7: Move above item2 (tomato2) on the table ---
        try:
            item2_pos = positions['item2']
        except KeyError:
            print("[Error] Could not find 'item2' in object positions.")
            shutdown_environment(env)
            return
        approach_item2_pos = np.array(item2_pos) + np.array([0, 0, 0.10])
        print("[Plan] Step 7: Move above item2 (tomato2)")
        obs, reward, done = move(env, task, target_pos=approach_item2_pos)
        if done:
            print("[Task] Task ended after move above item2!")
            return

        # --- Step 8: Pick item2 (tomato2) ---
        print("[Plan] Step 8: Pick item2 (tomato2)")
        obs, reward, done = pick(env, task, target_pos=np.array(item2_pos), approach_distance=0.10, approach_axis='-z')
        if done:
            print("[Task] Task ended after pick item2!")
            return

        # --- Step 9: Place item2 on the plate ---
        print("[Plan] Step 9a: Move above plate")
        obs, reward, done = move(env, task, target_pos=approach_plate_pos)
        if done:
            print("[Task] Task ended after move above plate (2nd time)!")
            return
        print("[Plan] Step 9b: Place item2 on plate")
        obs, reward, done = place(env, task, target_pos=np.array(plate_pos), approach_distance=0.10, approach_axis='-z')
        if done:
            print("[Task] Task ended after place item2!")
            return

        print("[Task] All plan steps executed. Task should be complete.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
