# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Slide open any available drawer to access its contents, then toss the 2 tomatoes from the table onto the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list and observation:
        # 'item1', 'item2', 'item3' (assume item1 and item2 are tomatoes), 'plate'
        # Drawer positions: 'bottom_anchor_pos', 'bottom_side_pos', 'middle_anchor_pos', etc.

        # --- Step 1: Move gripper to the side position of the bottom drawer ---
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] Could not find 'bottom_side_pos' in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Step 1: Move to bottom drawer side position.")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Task ended after move to bottom_side_pos!")
            return

        # --- Step 2: Rotate gripper to 90 degrees (for drawer grasp) ---
        # Assume 90 deg = [x, y, z, w] quaternion for 90 deg about z axis
        # We'll use scipy to get the quaternion for 90 deg about z
        from scipy.spatial.transform import Rotation as R
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        print("[Plan] Step 2: Rotate gripper to 90 degrees.")
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        if done:
            print("[Task] Task ended after rotate to 90 deg!")
            return

        # --- Step 3: Move to anchor position of the bottom drawer (for grasping handle) ---
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Error] Could not find 'bottom_anchor_pos' in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Step 3: Move to bottom drawer anchor position.")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Task ended after move to bottom_anchor_pos!")
            return

        # --- Step 4: Pick (grasp) the drawer handle at anchor position ---
        # For pick, we use the anchor position as the target
        print("[Plan] Step 4: Pick (grasp) the bottom drawer handle.")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended after pick (drawer handle)!")
            return

        # --- Step 5: Pull the drawer open ---
        # Pull along the x axis (assume positive x is outwards)
        print("[Plan] Step 5: Pull the bottom drawer open.")
        obs, reward, done = pull(env, task, pull_distance=0.18, pull_axis='x')
        if done:
            print("[Task] Task ended after pull (drawer)!")
            return

        # --- Step 6: Move to item1 (tomato1) on the table ---
        try:
            item1_pos = positions['item1']
        except KeyError:
            print("[Error] Could not find 'item1' in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Step 6: Move to item1 (tomato1) position.")
        obs, reward, done = move(env, task, target_pos=item1_pos)
        if done:
            print("[Task] Task ended after move to item1!")
            return

        # --- Step 7: Rotate gripper to 0 degrees (upright) for picking tomato ---
        zero_deg_quat = R.from_euler('z', 0, degrees=True).as_quat()
        print("[Plan] Step 7: Rotate gripper to 0 degrees (upright).")
        obs, reward, done = rotate(env, task, target_quat=zero_deg_quat)
        if done:
            print("[Task] Task ended after rotate to 0 deg!")
            return

        # --- Step 8: Move above item1 (tomato1) for picking ---
        # Optionally, move slightly above the item before picking
        item1_above = np.array(item1_pos) + np.array([0, 0, 0.05])
        print("[Plan] Step 8: Move above item1 (tomato1) for picking.")
        obs, reward, done = move(env, task, target_pos=item1_above)
        if done:
            print("[Task] Task ended after move above item1!")
            return

        # --- Step 9: Pick item1 (tomato1) ---
        print("[Plan] Step 9: Pick item1 (tomato1).")
        obs, reward, done = pick(env, task, target_pos=item1_pos, approach_distance=0.10, approach_axis='-z')
        if done:
            print("[Task] Task ended after pick item1!")
            return

        # --- Step 10: Place item1 (tomato1) on the plate ---
        try:
            plate_pos = positions['plate']
        except KeyError:
            print("[Error] Could not find 'plate' in object positions.")
            shutdown_environment(env)
            return

        # Place above the plate
        plate_above = np.array(plate_pos) + np.array([0, 0, 0.05])
        print("[Plan] Step 10: Place item1 (tomato1) on the plate.")
        obs, reward, done = place(env, task, target_pos=plate_above, approach_distance=0.10, approach_axis='-z')
        if done:
            print("[Task] Task ended after place item1!")
            return

        # --- Step 11: Move to item2 (tomato2) on the table ---
        try:
            item2_pos = positions['item2']
        except KeyError:
            print("[Error] Could not find 'item2' in object positions.")
            shutdown_environment(env)
            return

        print("[Plan] Step 11: Move to item2 (tomato2) position.")
        obs, reward, done = move(env, task, target_pos=item2_pos)
        if done:
            print("[Task] Task ended after move to item2!")
            return

        # --- Step 12: Pick item2 (tomato2) ---
        print("[Plan] Step 12: Pick item2 (tomato2).")
        obs, reward, done = pick(env, task, target_pos=item2_pos, approach_distance=0.10, approach_axis='-z')
        if done:
            print("[Task] Task ended after pick item2!")
            return

        # --- Step 13: Place item2 (tomato2) on the plate ---
        print("[Plan] Step 13: Place item2 (tomato2) on the plate.")
        obs, reward, done = place(env, task, target_pos=plate_above, approach_distance=0.10, approach_axis='-z')
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()