import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: pick, place, move, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Pull open any unlocked drawer, then drop the 2 tomatoes onto the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object name mapping (from object list and observation)
        # We need to find:
        # - Two tomatoes (item1, item2)
        # - Plate (plate)
        # - Any unlocked drawer (bottom, middle, top) and their anchor/side positions

        # 1. Find an unlocked drawer
        # From the initial state, only 'bottom' is not locked
        unlocked_drawer = 'bottom'
        gripper_name = 'gripper'
        # Anchor and side positions for the bottom drawer
        anchor_pos_name = 'bottom_anchor_pos'
        side_pos_name = 'bottom_side_pos'
        # The joint position is not used for manipulation, but may be in object_positions
        # The gripper starts at 'nowhere-pos', so we need to move it to the side, then anchor

        # 2. Get positions for manipulation
        try:
            # Drawer handle positions
            side_pos = positions[side_pos_name]
            anchor_pos = positions[anchor_pos_name]
            # Tomatoes and plate
            tomato1_name = 'item1'
            tomato2_name = 'item2'
            tomato1_pos = positions[tomato1_name]
            tomato2_pos = positions[tomato2_name]
            plate_pos = positions['plate']
        except KeyError as e:
            print(f"[Error] Missing object position for: {e}")
            shutdown_environment(env)
            return

        # === PLAN EXECUTION ===
        # Step 1: Pull open any unlocked drawer (bottom)
        #   - Move gripper to side position (side_pos)
        #   - Rotate gripper to 90 deg (if needed)
        #   - Move to anchor position (anchor_pos)
        #   - Pick drawer handle (simulate grasp)
        #   - Pull (along x axis, positive direction, e.g., 0.15m)
        #   - (We assume the pull skill is sufficient for the drawer to open)

        # Move to side position above the drawer
        print("[Task] Moving gripper to side position of drawer...")
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Task] Task ended unexpectedly after move to side position!")
            return

        # Rotate gripper to 90 degrees (if not already)
        # We'll try to get the quaternion for 90 deg about z axis
        from scipy.spatial.transform import Rotation as R
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw
        # The skill expects [x, y, z, w]
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        if done:
            print("[Task] Task ended unexpectedly after rotate!")
            return

        # Move to anchor position (where the handle is)
        print("[Task] Moving gripper to anchor position of drawer...")
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        if done:
            print("[Task] Task ended unexpectedly after move to anchor position!")
            return

        # Simulate grasping the drawer handle (close gripper)
        print("[Task] Grasping drawer handle (pick)...")
        # Use pick skill at anchor_pos (simulate grasping the handle)
        obs, reward, done = pick(env, task, target_pos=anchor_pos, approach_distance=0.05, approach_axis='z')
        if done:
            print("[Task] Task ended unexpectedly after pick drawer handle!")
            return

        # Pull the drawer open (along x axis, positive direction)
        print("[Task] Pulling the drawer open...")
        pull_distance = 0.15  # 15cm is typical for opening a drawer
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Task] Task ended unexpectedly after pull!")
            return

        # Step 2: Pick up tomato1 and place on plate
        print(f"[Task] Picking up {tomato1_name}...")
        obs, reward, done = pick(env, task, target_pos=tomato1_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print(f"[Task] Task ended unexpectedly after picking {tomato1_name}!")
            return

        print(f"[Task] Placing {tomato1_name} on plate...")
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print(f"[Task] Task ended unexpectedly after placing {tomato1_name}!")
            return

        # Step 3: Pick up tomato2 and place on plate
        print(f"[Task] Picking up {tomato2_name}...")
        obs, reward, done = pick(env, task, target_pos=tomato2_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print(f"[Task] Task ended unexpectedly after picking {tomato2_name}!")
            return

        print(f"[Task] Placing {tomato2_name} on plate...")
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print(f"[Task] Task ended unexpectedly after placing {tomato2_name}!")
            return

        print("[Task] Task completed successfully! Reward:", reward)

    except Exception as e:
        print(f"[Exception] {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()