import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Map object names to positions for convenience
        # The object list includes:
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos', 'waypoint1',
        # 'item1', 'item2', 'item3', 'plate'
        #
        # For the plan, we will:
        # 1. Open any available drawer (bottom, middle, or top). We'll choose 'bottom' (unlocked).
        # 2. Toss the 2 tomatoes (item1, item2) from the table onto the plate.

        # Defensive: Check all required keys exist
        required_keys = [
            'bottom_side_pos', 'bottom_anchor_pos', 'bottom_joint_pos',
            'item1', 'item2', 'plate'
        ]
        for k in required_keys:
            if k not in positions:
                raise RuntimeError(f"Required object position '{k}' not found in get_object_positions().")

        # Plan step mapping (see specification):
        # 1. move: Move gripper to bottom_side_pos (side of bottom drawer)
        # 2. rotate: Rotate gripper to 90 degrees (for grasping drawer handle)
        # 3. move: Move gripper to bottom_anchor_pos (handle position)
        # 4. pick: Grasp the drawer handle at anchor position
        # 5. pull: Pull the drawer open (along x axis, e.g. +x, by a fixed distance)
        # 6. move: Move gripper away from drawer (e.g. to waypoint1 or above table)
        # 7. rotate: Rotate gripper back to 0 degrees (for object pick)
        # 8. move: Move gripper above item1
        # 9. pick: Pick item1
        # 10. place: Place item1 on plate
        # 11. move: Move gripper above item2
        # 12. pick: Pick item2
        # 13. place: Place item2 on plate

        # Step 1: Move to bottom_side_pos
        print("[Step 1] Move to bottom drawer side position")
        obs, reward, done = move(
            env, task,
            target_pos=np.array(positions['bottom_side_pos'])
        )
        if done:
            print("[Task] Task ended after move to drawer side!")
            return

        # Step 2: Rotate to 90 degrees (drawer grasp orientation)
        print("[Step 2] Rotate gripper to 90 degrees")
        # We'll assume 90 deg is [x, y, z, w] quaternion for 90 deg about z axis
        # If positions has 'bottom_joint_pos' or similar, use that for orientation reference
        # For RLBench, 90 deg about z is [0, 0, sqrt(0.5), sqrt(0.5)]
        ninety_deg_quat = np.array([0, 0, np.sqrt(0.5), np.sqrt(0.5)])
        obs, reward, done = rotate(
            env, task,
            target_quat=ninety_deg_quat
        )
        if done:
            print("[Task] Task ended after rotate!")
            return

        # Step 3: Move to bottom_anchor_pos (drawer handle)
        print("[Step 3] Move to bottom drawer anchor (handle) position")
        obs, reward, done = move(
            env, task,
            target_pos=np.array(positions['bottom_anchor_pos'])
        )
        if done:
            print("[Task] Task ended after move to drawer anchor!")
            return

        # Step 4: Pick (grasp) the drawer handle
        print("[Step 4] Pick (grasp) the drawer handle")
        obs, reward, done = pick(
            env, task,
            target_pos=np.array(positions['bottom_anchor_pos']),
            approach_distance=0.10,  # Slightly closer approach for handle
            approach_axis='z'
        )
        if done:
            print("[Task] Task ended after picking drawer handle!")
            return

        # Step 5: Pull the drawer open (along +x, e.g. 0.15m)
        print("[Step 5] Pull the drawer open")
        pull_distance = 0.15  # meters, adjust as needed for your environment
        obs, reward, done = pull(
            env, task,
            pull_distance=pull_distance,
            pull_axis='x'
        )
        if done:
            print("[Task] Task ended after pulling drawer!")
            return

        # Step 6: Move gripper away from drawer (to waypoint1 if available, else above table)
        print("[Step 6] Move gripper away from drawer")
        if 'waypoint1' in positions:
            move_target = np.array(positions['waypoint1'])
        else:
            # Fallback: move above the table (e.g. above item1)
            move_target = np.array(positions['item1']) + np.array([0, 0, 0.15])
        obs, reward, done = move(
            env, task,
            target_pos=move_target
        )
        if done:
            print("[Task] Task ended after moving away from drawer!")
            return

        # Step 7: Rotate gripper back to 0 degrees (for object pick)
        print("[Step 7] Rotate gripper to 0 degrees")
        zero_deg_quat = np.array([0, 0, 0, 1])
        obs, reward, done = rotate(
            env, task,
            target_quat=zero_deg_quat
        )
        if done:
            print("[Task] Task ended after rotate to 0 deg!")
            return

        # Step 8: Move above item1
        print("[Step 8] Move above item1")
        item1_pos = np.array(positions['item1'])
        above_item1 = item1_pos + np.array([0, 0, 0.10])
        obs, reward, done = move(
            env, task,
            target_pos=above_item1
        )
        if done:
            print("[Task] Task ended after move above item1!")
            return

        # Step 9: Pick item1
        print("[Step 9] Pick item1")
        obs, reward, done = pick(
            env, task,
            target_pos=item1_pos,
            approach_distance=0.10,
            approach_axis='z'
        )
        if done:
            print("[Task] Task ended after picking item1!")
            return

        # Step 10: Place item1 on plate
        print("[Step 10] Place item1 on plate")
        plate_pos = np.array(positions['plate'])
        above_plate = plate_pos + np.array([0, 0, 0.10])
        obs, reward, done = place(
            env, task,
            target_pos=plate_pos,
            approach_distance=0.10,
            approach_axis='z'
        )
        if done:
            print("[Task] Task ended after placing item1!")
            return

        # Step 11: Move above item2
        print("[Step 11] Move above item2")
        item2_pos = np.array(positions['item2'])
        above_item2 = item2_pos + np.array([0, 0, 0.10])
        obs, reward, done = move(
            env, task,
            target_pos=above_item2
        )
        if done:
            print("[Task] Task ended after move above item2!")
            return

        # Step 12: Pick item2
        print("[Step 12] Pick item2")
        obs, reward, done = pick(
            env, task,
            target_pos=item2_pos,
            approach_distance=0.10,
            approach_axis='z'
        )
        if done:
            print("[Task] Task ended after picking item2!")
            return

        # Step 13: Place item2 on plate
        print("[Step 13] Place item2 on plate")
        obs, reward, done = place(
            env, task,
            target_pos=plate_pos,
            approach_distance=0.10,
            approach_axis='z'
        )
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()