# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # --- PLAN EXECUTION (according to the provided plan/specification) ---

        # Step 1: rotate(gripper, zero_deg, ninety_deg)
        # We assume the gripper's orientation for 'zero_deg' and 'ninety_deg' are provided in positions or can be constructed.
        # For this example, we assume positions['gripper_ninety_deg_quat'] exists, otherwise fallback to a default.
        try:
            # Try to get the target quaternion for ninety_deg
            target_quat = positions.get('gripper_ninety_deg_quat', None)
            if target_quat is None:
                # Fallback: rotate 90 deg around z axis from current orientation
                obs = task.get_observation()
                current_quat = obs.gripper_pose[3:7]
                from scipy.spatial.transform import Rotation as R
                r = R.from_quat(current_quat)
                ninety_rot = R.from_euler('z', 90, degrees=True)
                target_quat = (ninety_rot * r).as_quat()
            print("[Plan] Step 1: Rotating gripper to ninety_deg")
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[ERROR] Step 1 (rotate): {e}")
            shutdown_environment(env)
            return

        # Step 2: move(gripper, side-pos-bottom)
        try:
            target_pos = positions.get('bottom_side_pos', None)
            if target_pos is None:
                raise KeyError("Position for 'bottom_side_pos' not found in object_positions.")
            print("[Plan] Step 2: Moving gripper to side-pos-bottom")
            obs, reward, done = move(env, task, target_pos)
            if done:
                print("[Task] Task ended during move to side-pos-bottom!")
                return
        except Exception as e:
            print(f"[ERROR] Step 2 (move to side-pos-bottom): {e}")
            shutdown_environment(env)
            return

        # Step 3: move(gripper, anchor-pos-bottom)
        try:
            target_pos = positions.get('bottom_anchor_pos', None)
            if target_pos is None:
                raise KeyError("Position for 'bottom_anchor_pos' not found in object_positions.")
            print("[Plan] Step 3: Moving gripper to anchor-pos-bottom")
            obs, reward, done = move(env, task, target_pos)
            if done:
                print("[Task] Task ended during move to anchor-pos-bottom!")
                return
        except Exception as e:
            print(f"[ERROR] Step 3 (move to anchor-pos-bottom): {e}")
            shutdown_environment(env)
            return

        # Step 4: pick(gripper, bottom)
        # For picking the drawer handle, we use the anchor position as the target
        try:
            target_pos = positions.get('bottom_anchor_pos', None)
            if target_pos is None:
                raise KeyError("Position for 'bottom_anchor_pos' not found in object_positions.")
            print("[Plan] Step 4: Picking the bottom drawer handle")
            obs, reward, done = pick(env, task, target_pos, approach_distance=0.10, approach_axis='z')
            if done:
                print("[Task] Task ended during pick of drawer handle!")
                return
        except Exception as e:
            print(f"[ERROR] Step 4 (pick drawer handle): {e}")
            shutdown_environment(env)
            return

        # Step 5: pull(gripper, bottom)
        # For pulling, we assume the drawer pulls along the x axis by a certain distance (e.g., 0.15m)
        try:
            print("[Plan] Step 5: Pulling the bottom drawer")
            obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='x')
            if done:
                print("[Task] Task ended during pull of drawer!")
                return
        except Exception as e:
            print(f"[ERROR] Step 5 (pull drawer): {e}")
            shutdown_environment(env)
            return

        # Step 6: pick(tomato1, table)
        try:
            tomato1_pos = positions.get('tomato1', None)
            if tomato1_pos is None:
                raise KeyError("Position for 'tomato1' not found in object_positions.")
            print("[Plan] Step 6: Picking up tomato1 from table")
            obs, reward, done = pick(env, task, tomato1_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick of tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] Step 6 (pick tomato1): {e}")
            shutdown_environment(env)
            return

        # Step 7: place(tomato1, plate)
        try:
            plate_pos = positions.get('plate', None)
            if plate_pos is None:
                raise KeyError("Position for 'plate' not found in object_positions.")
            print("[Plan] Step 7: Placing tomato1 on plate")
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during place of tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] Step 7 (place tomato1): {e}")
            shutdown_environment(env)
            return

        # Step 8: pick(tomato2, table)
        try:
            tomato2_pos = positions.get('tomato2', None)
            if tomato2_pos is None:
                raise KeyError("Position for 'tomato2' not found in object_positions.")
            print("[Plan] Step 8: Picking up tomato2 from table")
            obs, reward, done = pick(env, task, tomato2_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick of tomato2!")
                return
        except Exception as e:
            print(f"[ERROR] Step 8 (pick tomato2): {e}")
            shutdown_environment(env)
            return

        # Step 9: place(tomato2, plate)
        try:
            plate_pos = positions.get('plate', None)
            if plate_pos is None:
                raise KeyError("Position for 'plate' not found in object_positions.")
            print("[Plan] Step 9: Placing tomato2 on plate")
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during place of tomato2!")
                return
        except Exception as e:
            print(f"[ERROR] Step 9 (place tomato2): {e}")
            shutdown_environment(env)
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
