import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === PLAN EXECUTION ===
        # Goal: Pull open any drawer that is not locked, then drop the 2 tomatoes onto the plate.

        # 1. Find an unlocked drawer (bottom is not locked in the initial state)
        # We'll use the bottom drawer for the pull action.
        # The plan is:
        #   1. Rotate gripper to 90 degrees (vertical, for side approach)
        #   2. Move to the side position of the bottom drawer
        #   3. Move to the anchor position of the bottom drawer
        #   4. Move to the handle (anchor) position (redundant, but matches plan steps)
        #   5. Pull the drawer open
        #   6. Pick tomato1 from table
        #   7. Place tomato1 on plate
        #   8. Pick tomato2 from table
        #   9. Place tomato2 on plate

        # --- Step 1: Rotate gripper to 90 degrees (vertical) ---
        try:
            # Get current gripper pose
            obs = task.get_observation()
            # Target orientation: 90 deg about z axis (assuming 'zero_deg' is default, 'ninety_deg' is vertical)
            # We'll rotate about z axis by 90 degrees
            from scipy.spatial.transform import Rotation as R
            start_quat = obs.gripper_pose[3:7]
            # Compose a 90 degree rotation about z
            rot_90_z = R.from_euler('z', 90, degrees=True)
            target_quat = rot_90_z.as_quat()  # xyzw
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended after rotate!")
                return
        except Exception as e:
            print(f"[ERROR] Step 1 (rotate) failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 2: Move to side position of bottom drawer ---
        try:
            side_pos = positions.get('bottom_side_pos', None)
            if side_pos is None:
                raise KeyError("bottom_side_pos not found in positions.")
            obs, reward, done = move(env, task, side_pos)
            if done:
                print("[Task] Task ended after move to side position!")
                return
        except Exception as e:
            print(f"[ERROR] Step 2 (move to side) failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 3: Move to anchor position of bottom drawer ---
        try:
            anchor_pos = positions.get('bottom_anchor_pos', None)
            if anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in positions.")
            obs, reward, done = move(env, task, anchor_pos)
            if done:
                print("[Task] Task ended after move to anchor position!")
                return
        except Exception as e:
            print(f"[ERROR] Step 3 (move to anchor) failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 4: Move to anchor position again (redundant, but matches plan) ---
        try:
            obs, reward, done = move(env, task, anchor_pos)
            if done:
                print("[Task] Task ended after move to anchor (repeat)!")
                return
        except Exception as e:
            print(f"[ERROR] Step 4 (move to anchor repeat) failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 5: Pull the drawer open ---
        try:
            # Pull along x axis (assume drawer opens along +x, adjust if needed)
            pull_distance = 0.15  # 15cm, adjust as needed for your environment
            obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
            if done:
                print("[Task] Task ended after pull!")
                return
        except Exception as e:
            print(f"[ERROR] Step 5 (pull) failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 6: Pick tomato1 from table ---
        try:
            tomato1_pos = positions.get('tomato1', None)
            if tomato1_pos is None:
                raise KeyError("tomato1 not found in positions.")
            obs, reward, done = pick(env, task, tomato1_pos)
            if done:
                print("[Task] Task ended after pick tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] Step 6 (pick tomato1) failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 7: Place tomato1 on plate ---
        try:
            plate_pos = positions.get('plate', None)
            if plate_pos is None:
                raise KeyError("plate not found in positions.")
            obs, reward, done = place(env, task, plate_pos)
            if done:
                print("[Task] Task ended after place tomato1!")
                return
        except Exception as e:
            print(f"[ERROR] Step 7 (place tomato1) failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 8: Pick tomato2 from table ---
        try:
            tomato2_pos = positions.get('tomato2', None)
            if tomato2_pos is None:
                raise KeyError("tomato2 not found in positions.")
            obs, reward, done = pick(env, task, tomato2_pos)
            if done:
                print("[Task] Task ended after pick tomato2!")
                return
        except Exception as e:
            print(f"[ERROR] Step 8 (pick tomato2) failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 9: Place tomato2 on plate ---
        try:
            obs, reward, done = place(env, task, plate_pos)
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[ERROR] Step 9 (place tomato2) failed: {e}")
            shutdown_environment(env)
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()