import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer fully, then pick up all tomatoes and place them on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object keys: 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos', 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos', 'top_anchor_pos', 'top_joint_pos', 'top_side_pos', 'waypoint1', 'rubbish', 'tomato1', 'tomato2', 'plate'

        # --- Plan: Open a drawer fully (choose 'bottom' drawer for simplicity) ---
        # 1. Rotate gripper to 90 degrees (vertical for side approach)
        # 2. Move to side position of bottom drawer
        # 3. Move to anchor position of bottom drawer
        # 4. Pick the drawer handle (anchor)
        # 5. Pull the drawer open (along +x axis, e.g. 0.18m)
        # 6. Pick tomato1 from table
        # 7. Place tomato1 on plate
        # 8. Pick tomato2 from table
        # 9. Place tomato2 on plate

        # --- Step 1: Rotate gripper to 90 degrees (vertical) ---
        try:
            # Assume 90 deg = [x, y, z, w] quaternion for vertical orientation
            # If object_positions provides 'bottom_side_pos', use its orientation as reference
            # Otherwise, use a hardcoded quaternion for 90 deg about x (RLBench: [0.707, 0, 0, 0.707])
            ninety_deg_quat = np.array([0.7071068, 0, 0, 0.7071068])
            obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
            if done:
                print("[Task] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[Error] Rotate failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 2: Move to side position of bottom drawer ---
        try:
            side_pos = positions.get('bottom_side_pos', None)
            if side_pos is None:
                raise KeyError("bottom_side_pos not found in object positions.")
            obs, reward, done = move(env, task, target_pos=side_pos)
            if done:
                print("[Task] Task ended during move to side position!")
                return
        except Exception as e:
            print(f"[Error] Move to side position failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 3: Move to anchor position of bottom drawer ---
        try:
            anchor_pos = positions.get('bottom_anchor_pos', None)
            if anchor_pos is None:
                raise KeyError("bottom_anchor_pos not found in object positions.")
            obs, reward, done = move(env, task, target_pos=anchor_pos)
            if done:
                print("[Task] Task ended during move to anchor position!")
                return
        except Exception as e:
            print(f"[Error] Move to anchor position failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 4: Pick the drawer handle (anchor) ---
        try:
            # Approach from -z (downward) to avoid collision
            obs, reward, done = pick(env, task, target_pos=anchor_pos, approach_distance=0.10, approach_axis='-z')
            if done:
                print("[Task] Task ended during pick drawer handle!")
                return
        except Exception as e:
            print(f"[Error] Pick drawer handle failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 5: Pull the drawer open (along +x axis) ---
        try:
            # Pull distance: 0.18m (tune as needed for full open)
            obs, reward, done = pull(env, task, pull_distance=0.18, pull_axis='x')
            if done:
                print("[Task] Task ended during pull drawer!")
                return
        except Exception as e:
            print(f"[Error] Pull drawer failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 6: Pick tomato1 from table ---
        try:
            tomato1_pos = positions.get('tomato1', None)
            if tomato1_pos is None:
                raise KeyError("tomato1 not found in object positions.")
            obs, reward, done = pick(env, task, target_pos=tomato1_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick tomato1!")
                return
        except Exception as e:
            print(f"[Error] Pick tomato1 failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 7: Place tomato1 on plate ---
        try:
            plate_pos = positions.get('plate', None)
            if plate_pos is None:
                raise KeyError("plate not found in object positions.")
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during place tomato1!")
                return
        except Exception as e:
            print(f"[Error] Place tomato1 failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 8: Pick tomato2 from table ---
        try:
            tomato2_pos = positions.get('tomato2', None)
            if tomato2_pos is None:
                raise KeyError("tomato2 not found in object positions.")
            obs, reward, done = pick(env, task, target_pos=tomato2_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick tomato2!")
                return
        except Exception as e:
            print(f"[Error] Pick tomato2 failed: {e}")
            shutdown_environment(env)
            return

        # --- Step 9: Place tomato2 on plate ---
        try:
            obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task completed successfully! Reward:", reward)
            else:
                print("[Task] Task not completed yet (done=False).")
        except Exception as e:
            print(f"[Error] Place tomato2 failed: {e}")
            shutdown_environment(env)
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()