import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer fully, then pick up all tomatoes and place them on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object keys: 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos', 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos', 'top_anchor_pos', 'top_joint_pos', 'top_side_pos', 'waypoint1', 'rubbish', 'tomato1', 'tomato2', 'plate'

        # --- Plan: Open a drawer fully, then pick up all tomatoes and place them on the plate ---

        # 1. Choose a drawer to open. We'll use the 'bottom' drawer.
        # 2. Move gripper to the side position of the bottom drawer.
        # 3. Rotate gripper to 90 degrees (for grasping the handle).
        # 4. Move gripper to the anchor position (handle) of the bottom drawer.
        # 5. Pick the drawer handle (simulate grasp).
        # 6. Pull the drawer open (along x axis).
        # 7. Move gripper away from the drawer to a safe position.
        # 8. Pick up tomato1 from the table, place on plate.
        # 9. Pick up tomato2 from the table, place on plate.

        # --- Step 1: Move to side position of bottom drawer ---
        try:
            bottom_side_pos = positions['bottom_side_pos']
        except KeyError:
            print("[Error] 'bottom_side_pos' not found in object positions.")
            shutdown_environment(env)
            return

        print("[Task] Moving to bottom drawer side position:", bottom_side_pos)
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Task ended unexpectedly after move to side position.")
            return

        # --- Step 2: Rotate gripper to 90 degrees (for handle approach) ---
        # We'll use a quaternion for 90 deg rotation about z axis.
        # RLBench uses [x, y, z, w] (xyzw) format.
        from scipy.spatial.transform import Rotation as R
        quat_ninety_deg = R.from_euler('z', 90, degrees=True).as_quat()  # [x, y, z, w]
        print("[Task] Rotating gripper to 90 degrees (about z axis):", quat_ninety_deg)
        obs, reward, done = rotate(env, task, target_quat=quat_ninety_deg)
        if done:
            print("[Task] Task ended unexpectedly after rotate.")
            return

        # --- Step 3: Move to anchor (handle) position of bottom drawer ---
        try:
            bottom_anchor_pos = positions['bottom_anchor_pos']
        except KeyError:
            print("[Error] 'bottom_anchor_pos' not found in object positions.")
            shutdown_environment(env)
            return

        print("[Task] Moving to bottom drawer anchor (handle) position:", bottom_anchor_pos)
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Task ended unexpectedly after move to anchor position.")
            return

        # --- Step 4: Pick the drawer handle (simulate grasp) ---
        print("[Task] Grasping the drawer handle (pick on anchor position).")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos, approach_distance=0.10, approach_axis='z')
        if done:
            print("[Task] Task ended unexpectedly after picking drawer handle.")
            return

        # --- Step 5: Pull the drawer open (along x axis) ---
        # We'll pull by a fixed distance (e.g., 0.20 meters along x axis)
        pull_distance = 0.20
        print(f"[Task] Pulling the drawer open by {pull_distance} meters along x axis.")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Task] Task ended unexpectedly after pulling drawer.")
            return

        # --- Step 6: Move gripper away from drawer to a safe position (optional) ---
        # We'll move back to the side position to avoid collision.
        print("[Task] Moving gripper away from drawer to side position.")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Task ended unexpectedly after moving away from drawer.")
            return

        # --- Step 7: Pick up tomato1 from table and place on plate ---
        try:
            tomato1_pos = positions['tomato1']
            plate_pos = positions['plate']
        except KeyError as e:
            print(f"[Error] '{e.args[0]}' not found in object positions.")
            shutdown_environment(env)
            return

        print("[Task] Picking up tomato1 from table at:", tomato1_pos)
        obs, reward, done = pick(env, task, target_pos=tomato1_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended unexpectedly after picking tomato1.")
            return

        print("[Task] Placing tomato1 on plate at:", plate_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended unexpectedly after placing tomato1.")
            return

        # --- Step 8: Pick up tomato2 from table and place on plate ---
        try:
            tomato2_pos = positions['tomato2']
        except KeyError:
            print("[Error] 'tomato2' not found in object positions.")
            shutdown_environment(env)
            return

        print("[Task] Picking up tomato2 from table at:", tomato2_pos)
        obs, reward, done = pick(env, task, target_pos=tomato2_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended unexpectedly after picking tomato2.")
            return

        print("[Task] Placing tomato2 on plate at:", plate_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15, approach_axis='z')
        if done:
            print("[Task] Task ended after placing tomato2. Task complete!")
            return

        print("[Task] All tomatoes placed on plate. Task complete!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()