# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open a drawer and toss the 2 tomatoes from the table onto the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object keys expected in positions:
        # 'tomato1', 'tomato2', 'plate', 'bottom_anchor_pos', 'bottom_side_pos', 'middle_anchor_pos', etc.

        # --- Step 1: Rotate gripper to 90 degrees (for drawer manipulation) ---
        # We'll use the bottom drawer (unlocked) for simplicity.
        # Find the quaternion for 90 degrees rotation about z axis.
        from scipy.spatial.transform import Rotation as R
        # RLBench uses [x, y, z, w] convention
        ninety_deg_quat = R.from_euler('z', 90, degrees=True).as_quat()  # [x, y, z, w]
        # Ensure it's a numpy array
        ninety_deg_quat = np.array(ninety_deg_quat)
        print("[Task] Rotating gripper to 90 degrees (z axis)")
        obs, reward, done = rotate(env, task, target_quat=ninety_deg_quat)
        if done:
            print("[Task] Task ended during rotate!")
            return

        # --- Step 2: Move to the side position of the bottom drawer ---
        bottom_side_pos = positions.get('bottom_side_pos', None)
        if bottom_side_pos is None:
            raise RuntimeError("bottom_side_pos not found in object positions.")
        print("[Task] Moving to bottom drawer side position:", bottom_side_pos)
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Task ended during move to side position!")
            return

        # --- Step 3: Move to the anchor position of the bottom drawer ---
        bottom_anchor_pos = positions.get('bottom_anchor_pos', None)
        if bottom_anchor_pos is None:
            raise RuntimeError("bottom_anchor_pos not found in object positions.")
        print("[Task] Moving to bottom drawer anchor position:", bottom_anchor_pos)
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Task ended during move to anchor position!")
            return

        # --- Step 4: (Implicitly grip the drawer handle by being at anchor pos with gripper closed) ---
        # The skill primitives do not have a separate "pick-drawer" function; we assume the robot is ready to pull.

        # --- Step 5: Pull the drawer open ---
        # Pull along the x axis (assume positive x opens the drawer), distance ~0.15m
        pull_distance = 0.15
        print("[Task] Pulling the bottom drawer open (along +x, 0.15m)")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Task] Task ended during pull!")
            return

        # --- Step 6: Pick up tomato1 from the table ---
        tomato1_pos = positions.get('tomato1', None)
        if tomato1_pos is None:
            raise RuntimeError("tomato1 not found in object positions.")
        print("[Task] Picking up tomato1 at:", tomato1_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato1_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after picking tomato1!")
            return

        # --- Step 7: Place tomato1 onto the plate ---
        plate_pos = positions.get('plate', None)
        if plate_pos is None:
            raise RuntimeError("plate not found in object positions.")
        print("[Task] Placing tomato1 onto plate at:", plate_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after placing tomato1!")
            return

        # --- Step 8: Pick up tomato2 from the table ---
        tomato2_pos = positions.get('tomato2', None)
        if tomato2_pos is None:
            raise RuntimeError("tomato2 not found in object positions.")
        print("[Task] Picking up tomato2 at:", tomato2_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato2_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after picking tomato2!")
            return

        # --- Step 9: Place tomato2 onto the plate ---
        print("[Task] Placing tomato2 onto plate at:", plate_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after placing tomato2! Reward:", reward)
        else:
            print("[Task] Task completed, but not signaled as done. Reward:", reward)

    except Exception as e:
        print("[Task] Exception occurred:", str(e))
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()