# run_skeleton_task.py

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use only the provided skills

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list:
        # [
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos',
        # 'item1', 'item2', 'item3', 'plate'
        # ]

        # === PLAN EXECUTION ===
        # Goal: Unlock the cabinet by opening an accessible drawer, then transfer the 2 tomatoes onto the plate, leaving any other items undisturbed.

        # Assumptions:
        # - "item1" and "item2" are the tomatoes (since only 2 tomatoes are mentioned, and 3 items are present).
        # - "item3" is a non-tomato item and should not be moved.
        # - "plate" is the destination for the tomatoes.
        # - The bottom drawer is accessible and can be opened.
        # - The gripper starts at "nowhere-pos" and must be moved/rotated to interact with the drawer.

        # Step 1: Rotate gripper to 90 degrees (required for side approach to drawer)
        try:
            # Get current gripper pose
            obs = task.get_observation()
            start_quat = obs.gripper_pose[3:7]
            # Target: 90 degrees about z axis (assuming 'ninety_deg' means 90 deg about z)
            # We'll use scipy to get the quaternion for 90 deg about z
            from scipy.spatial.transform import Rotation as R
            target_rot = R.from_euler('z', 90, degrees=True)
            target_quat = target_rot.as_quat()  # xyzw
            obs, reward, done = rotate(env, task, target_quat)
            if done:
                print("[Task] Task ended during rotate!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during rotate: {e}")
            shutdown_environment(env)
            return

        # Step 2: Move gripper to the side position of the bottom drawer
        try:
            bottom_side_pos = positions.get('bottom_side_pos')
            if bottom_side_pos is None:
                raise ValueError("bottom_side_pos not found in positions.")
            obs, reward, done = move(env, task, bottom_side_pos)
            if done:
                print("[Task] Task ended during move to bottom_side_pos!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during move to bottom_side_pos: {e}")
            shutdown_environment(env)
            return

        # Step 3: Pull the bottom drawer open
        try:
            # For pull, we need to know the pull direction and distance.
            # Assume pulling along +x by 0.15m (typical drawer pull distance)
            pull_distance = 0.15
            pull_axis = 'x'
            obs, reward, done = pull(env, task, pull_distance, pull_axis)
            if done:
                print("[Task] Task ended during pull!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pull: {e}")
            shutdown_environment(env)
            return

        # Step 4: Pick up the first tomato (item1) from the table
        try:
            item1_pos = positions.get('item1')
            if item1_pos is None:
                raise ValueError("item1 not found in positions.")
            obs, reward, done = pick(env, task, item1_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick of item1!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pick of item1: {e}")
            shutdown_environment(env)
            return

        # Step 5: Place the first tomato (item1) onto the plate
        try:
            plate_pos = positions.get('plate')
            if plate_pos is None:
                raise ValueError("plate not found in positions.")
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during place of item1!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during place of item1: {e}")
            shutdown_environment(env)
            return

        # Step 6: Pick up the second tomato (item2) from the table
        try:
            item2_pos = positions.get('item2')
            if item2_pos is None:
                raise ValueError("item2 not found in positions.")
            obs, reward, done = pick(env, task, item2_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during pick of item2!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during pick of item2: {e}")
            shutdown_environment(env)
            return

        # Step 7: Place the second tomato (item2) onto the plate
        try:
            obs, reward, done = place(env, task, plate_pos, approach_distance=0.15, approach_axis='z')
            if done:
                print("[Task] Task ended during place of item2!")
                return
        except Exception as e:
            print(f"[ERROR] Exception during place of item2: {e}")
            shutdown_environment(env)
            return

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
