import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Open one of the drawers in the cabinet, then put all tomatoes on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object name mapping (from object list and environment)
        # Drawers: bottom, middle, top
        # Items: item1, item2, item3 (assume these are tomatoes)
        # Plate: plate

        # For robustness, check for required objects
        required_objects = ['bottom_anchor_pos', 'bottom_side_pos', 'middle_anchor_pos', 'middle_side_pos', 'top_anchor_pos', 'top_side_pos', 'item1', 'item2', 'item3', 'plate']
        for obj in required_objects:
            if obj not in positions:
                raise RuntimeError(f"Required object '{obj}' not found in object_positions.")

        # Choose which drawer to open (bottom drawer for this plan)
        drawer_anchor_pos = positions['bottom_anchor_pos']
        drawer_side_pos = positions['bottom_side_pos']
        drawer_pull_axis = 'x'  # Assume drawer pulls along x axis (positive direction)
        drawer_pull_distance = 0.18  # Empirically reasonable pull distance (meters)
        drawer_handle_approach_axis = 'z'  # Approach handle from above

        # Plate position
        plate_pos = positions['plate']

        # Tomato objects and their positions
        tomato_names = ['item1', 'item2', 'item3']
        tomato_positions = [positions[name] for name in tomato_names]

        # === PLAN EXECUTION ===
        # [Frozen Code Start]
        obs, reward, done = move(
                env,
                task,
                target_pos=drawer_side_pos,
                max_steps=100,
                threshold=0.01,
                timeout=10.0
            )
        obs, reward, done = rotate(
            env,
            task,
            target_quat=quat_90_z,
            max_steps=100,
            threshold=0.05,
            timeout=10.0
        )
        obs, reward, done = move(
            env,
            task,
            target_pos=drawer_anchor_pos,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        obs, reward, done = pick(
            env,
            task,
            target_pos=drawer_anchor_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis=drawer_handle_approach_axis,
            timeout=10.0
        )
        obs, reward, done = pull(
            env,
            task,
            pull_distance=drawer_pull_distance,
            pull_axis=drawer_pull_axis,
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        obs, reward, done = move(
            env,
            task,
            target_pos=tomato_positions[0],
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        # [Frozen Code End]

        if done:
            print("[Task] Task ended after move to first tomato!")
            return

        # Step 7: Pick first tomato (item1)
        print("[Plan] Step 7: Pick first tomato (item1).")
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato_positions[0],
            approach_distance=0.12,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick item1!")
            return

        # Step 8: Place first tomato on plate
        print("[Plan] Step 8: Place first tomato on plate.")
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.12,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after place item1!")
            return

        # Step 9: Move to second tomato (item2)
        print("[Plan] Step 9: Move to second tomato (item2).")
        obs, reward, done = move(
            env,
            task,
            target_pos=tomato_positions[1],
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to item2!")
            return

        # Step 10: Pick second tomato (item2)
        print("[Plan] Step 10: Pick second tomato (item2).")
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato_positions[1],
            approach_distance=0.12,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick item2!")
            return

        # Step 11: Place second tomato on plate
        print("[Plan] Step 11: Place second tomato on plate.")
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.12,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after place item2!")
            return

        # Step 12: Move to third tomato (item3)
        print("[Plan] Step 12: Move to third tomato (item3).")
        obs, reward, done = move(
            env,
            task,
            target_pos=tomato_positions[2],
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after move to item3!")
            return

        # Step 13: Pick third tomato (item3)
        print("[Plan] Step 13: Pick third tomato (item3).")
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato_positions[2],
            approach_distance=0.12,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after pick item3!")
            return

        # Step 14: Place third tomato on plate
        print("[Plan] Step 14: Place third tomato on plate.")
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.12,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended after place item3!")
            return

        print("[Task] All tomatoes placed on plate. Task completed successfully!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()