import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Use provided skills: move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Task: Pull open any unlocked drawer, then place the 2 tomatoes onto the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Object names from object list
        # [
        # 'bottom_anchor_pos', 'bottom_joint_pos', 'bottom_side_pos',
        # 'middle_anchor_pos', 'middle_joint_pos', 'middle_side_pos',
        # 'top_anchor_pos', 'top_joint_pos', 'top_side_pos',
        # 'item1', 'item2', 'item3', 'plate'
        # ]
        # We'll treat item1 and item2 as the "tomatoes" (since the goal says "2 tomatoes")
        # and plate as the destination.
        # For the drawer, open any that is not locked. We'll choose 'bottom' drawer.

        # Defensive: check all required keys exist
        required_keys = [
            'bottom_anchor_pos', 'bottom_side_pos', 'item1', 'item2', 'plate'
        ]
        for k in required_keys:
            if k not in positions:
                raise RuntimeError(f"Required object position '{k}' not found in get_object_positions().")

        # Drawer positions
        bottom_anchor_pos = positions['bottom_anchor_pos']
        bottom_side_pos = positions['bottom_side_pos']
        # Items
        tomato1_pos = positions['item1']
        tomato2_pos = positions['item2']
        plate_pos = positions['plate']

        # === PLAN EXECUTION ===
        # Plan (from specification):
        # 1. rotate
        # 2. move
        # 3. move
        # 4. pick
        # 5. pull
        # 6. pick
        # 7. place
        # 8. pick
        # 9. place

        # [Frozen Code Start]
        obs, reward, done = move(
                env, task, target_pos=bottom_side_pos
            )
        # [Frozen Code End]
        if done:
            print("[Task] Task ended after move to drawer side!")
            return

        # Step 2: Rotate gripper to 90 degrees about z axis
        print("[Step 2] Rotate gripper to 90 degrees about z axis")
        # We'll assume the gripper's orientation is quaternion, and 90 deg about z is [0,0,sin(45°),cos(45°)]
        # RLBench uses [x, y, z, w] order for quaternions
        angle_rad = np.deg2rad(90)
        qz_90 = np.array([0, 0, np.sin(angle_rad/2), np.cos(angle_rad/2)])
        obs, reward, done = rotate(
            env, task, target_quat=qz_90
        )
        if done:
            print("[Task] Task ended after rotate!")
            return

        # Step 3: Move to anchor position of the bottom drawer (for grasping handle)
        print("[Step 3] Move to bottom_anchor_pos (drawer anchor)")
        obs, reward, done = move(
            env, task, target_pos=bottom_anchor_pos
        )
        if done:
            print("[Task] Task ended after move to drawer anchor!")
            return

        # Step 4: Move to bottom_anchor_pos again (ensure correct approach for pick)
        print("[Step 4] Move to bottom_anchor_pos (pre-pick approach)")
        obs, reward, done = move(
            env, task, target_pos=bottom_anchor_pos
        )
        if done:
            print("[Task] Task ended after move to drawer anchor (pre-pick)!")
            return

        # Step 5: Pull the drawer open (pull along x axis, e.g., 0.15m)
        print("[Step 5] Pull the drawer open along x axis")
        obs, reward, done = pull(
            env, task, pull_distance=0.15, pull_axis='x'
        )
        if done:
            print("[Task] Task ended after pulling drawer!")
            return

        # Step 6: Pick item1 (tomato1)
        print("[Step 6] Pick tomato1")
        obs, reward, done = pick(
            env, task, target_pos=tomato1_pos
        )
        if done:
            print("[Task] Task ended after picking tomato1!")
            return

        # Step 7: Place tomato1 on plate
        print("[Step 7] Place tomato1 on plate")
        obs, reward, done = place(
            env, task, target_pos=plate_pos
        )
        if done:
            print("[Task] Task ended after placing tomato1!")
            return

        # Step 8: Pick item2 (tomato2)
        print("[Step 8] Pick tomato2")
        obs, reward, done = pick(
            env, task, target_pos=tomato2_pos
        )
        if done:
            print("[Task] Task ended after picking tomato2!")
            return

        # Step 9: Place tomato2 on plate
        print("[Step 9] Place tomato2 on plate")
        obs, reward, done = place(
            env, task, target_pos=plate_pos
        )
        if done:
            print("[Task] Task completed after placing tomato2! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()